<?php
/**
 * @package    com_datsogallery
 * @author     Andrey Datso <support@datso.fr>
 * @copyright  (c) 2006 - 2020 Andrey Datso. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

class DatsogalleryModelCategory extends JModelList
{

  protected $_total      = null;
  protected $_pagination = null;

  public function getTable($type = 'Category', $prefix = 'DatsogalleryTable', $config = array())
  {
    $table = JTable::getInstance($type, $prefix, $config);
    return $table;
  }

  protected function populateState($ordering = null, $direction = null)
  {
    $app          = JFactory::getApplication();
    $settings     = dg::getConfig();
    $order_method = $settings->category_order;
    switch ($order_method) {
      case 'id':
        $orderby = 'a.id';
        break;

      case 'rid':
        $orderby = 'a.id DESC';
        break;

      case 'alpha':
        $orderby = 'a.title';
        break;

      case 'ralpha':
        $orderby = 'a.title DESC';
        break;

      case 'rhits':
        $orderby = 'a.hits';
        break;

      case 'hits':
        $orderby = 'a.hits DESC';
        break;

      case 'order':
        $orderby = 'a.ordering';
        break;

      case 'rorder':
        $orderby = 'a.ordering DESC';
        break;

      default:
        $orderby = 'a.ordering';
        break;
    }
    $params = JComponentHelper::getParams('com_datsogallery');
    if ($app->input->get('layout') == 'edit') {
      $id = $app->getUserState('com_datsogallery.edit.category.id');
    } else {
      $id = $app->input->get('catid');
      $app->setUserState('com_datsogallery.edit.category.id', $id);
    }
    $this->setState('category.id', $id);
    $limit = $app->input->get('limit', $settings->category_per_page, 'uint');
    $this->setState('list.limit', $limit);
    $limitstart = $app->input->get('limitstart', 0, 'uint');
    $this->setState('list.start', $limitstart);
    $this->setState('list.ordering', $orderby);
    $this->setState('filter.language', JLanguageMultilang::isEnabled());
    $this->setState('params', $params);
  }

  protected function getCategories()
  {
    $app          = JFactory::getApplication();
    $settings     = dg::getConfig();
    $order_method = $settings->categories_order;
    $id           = $this->getState('category.id');
    switch ($order_method) {
      case 'id':
        $orderby = 'a.id';
        break;

      case 'rid':
        $orderby = 'a.id DESC';
        break;

      case 'alpha':
        $orderby = 'a.title';
        break;

      case 'ralpha':
        $orderby = 'a.title DESC';
        break;

      case 'order':
        $orderby = 'a.ordering';
        break;

      case 'rorder':
        $orderby = 'a.ordering DESC';
        break;

      default:
        $orderby = 'a.ordering';
        break;
    }
    $user     = JFactory::getUser();
    $groups   = implode(',', $user->getAuthorisedViewLevels());
    $db       = $this->getDbo();
    $nullDate = $db->q($db->getNullDate());
    $date     = JFactory::getDate();
    $nowDate  = $db->q($date->toSql());
    $query    = $db->getQuery(true);
    $query->select($this->getState('list.select', 'a.id, a.parent_id, a.title, a.description, a.created_by, a.created_time, a.image_id, u.id AS pc_user_id, u.name AS pc_user_name, u.email AS pc_user_email'));
    $query->from($db->qn('#__datsogallery_categories', 'a'));
    $query->join('LEFT', $db->qn('#__users', 'u') . ' ON ' . $db->qn('a.created_by') . ' = ' . $db->qn('u.id'));
    $query->where($db->qn('a.parent_id') . ' = ' . $id);
    $query->where($db->qn('a.published') . ' = 1');
    $query->where($db->qn('a.approved') . ' = 1');
    $query->where($db->qn('a.access') . ' IN (' . $groups . ')');
    $query->where('(' . $db->qn('a.publish_up') . ' = ' . $nullDate . ' OR ' . $db->qn('a.publish_up') . ' <= ' . $nowDate . ')');
    $query->where('(' . $db->qn('a.publish_down') . ' = ' . $nullDate . ' OR ' . $db->qn('a.publish_down') . ' >= ' . $nowDate . ')');
    if (JLanguageMultilang::isEnabled()) {
      $query->where($db->qn('a.language') . ' IN (' . $db->q(JFactory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');
    }
    $query->order($orderby);
    $db->setQuery($query);
    //var_dump((string) $query->__toString());
    $categories = $db->loadObjectList();
    return $categories;
  }

  protected function getImages()
  {
    $app      = JFactory::getApplication();
    $settings = dg::getConfig();
    $user     = JFactory::getUser();
    $groups   = implode(',', $user->getAuthorisedViewLevels());
    $db       = $this->getDbo();
    $nullDate = $db->q($db->getNullDate());
    $date     = JFactory::getDate();
    $nowDate  = $db->q($date->toSql());
    $catid    = $this->getState('category.id');
    $query    = $db->getQuery(true);
    $query->select('a.id, a.catid, a.title, a.description, a.original, a.featured, a.hits, a.downloads, a.created_by, a.created, u.id AS user_id, u.name AS user_name, u.email AS user_email');
    $query->from($db->qn('#__datsogallery_images', 'a'));
    $query->where($db->qn('a.access') . ' IN (' . $groups . ')');
    $query->where($db->qn('a.state') . ' = 1');
    $query->where($db->qn('a.approved') . ' = 1');
    $query->where('(' . $db->qn('a.publish_up') . ' = ' . $nullDate . ' OR ' . $db->qn('a.publish_up') . ' <= ' . $nowDate . ')');
    $query->where('(' . $db->qn('a.publish_down') . ' = ' . $nullDate . ' OR ' . $db->qn('a.publish_down') . ' >= ' . $nowDate . ')');
    $query->join('LEFT', $db->qn('#__users', 'u') . ' ON ' . $db->qn('a.created_by') . ' = ' . $db->qn('u.id'));
    $query->join('LEFT', $db->qn('#__datsogallery_categories', 'c') . ' ON ' . $db->qn('a.catid') . ' = ' . $db->qn('c.id'));
    $query->where($db->qn('c.access') . ' IN (' . $groups . ')');
    $query->where($db->qn('c.published') . ' = 1');
    $query->where($db->qn('c.approved') . ' = 1');
    $query->where('(' . $db->qn('c.publish_up') . ' = ' . $nullDate . ' OR ' . $db->qn('c.publish_up') . ' <= ' . $nowDate . ')');
    $query->where('(' . $db->qn('c.publish_down') . ' = ' . $nullDate . ' OR ' . $db->qn('c.publish_down') . ' >= ' . $nowDate . ')');
    $query->where($db->qn('a.catid') . ' = ' . $catid);
    if (JLanguageMultilang::isEnabled()) {
      $query->where($db->qn('a.language') . ' IN (' . $db->q(JFactory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');
      $query->where($db->qn('c.language') . ' IN (' . $db->q(JFactory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');
    }
    $query->order($db->escape($this->getState('list.ordering')));
    $db->setQuery($query);
    //var_dump((string) $query->__toString());
    $images = $db->loadObjectList();
    return $images;
  }

  public function getItems()
  {
    $results    = array();
    $categories = $this->getCategories();
    $rows       = array();
    if (count($categories) > 0) {
      foreach ($categories as $key => $item) {
        $categories[$key]->view = "category";
        $categories[$key]->href = JRoute::_("index.php?option=com_datsogallery&amp;view=category&amp;catid=" . $item->id . dg::menuid($item->id));
      }
    }
    $rows[] = $categories;
    $images = $this->getImages();
    if (count($images) > 0) {
      foreach ($images as $key => $item) {
        $images[$key]->view = "image";
        $images[$key]->href = JRoute::_('index.php?option=com_datsogallery&amp;view=image&amp;catid=' . $item->catid . '&amp;id=' . $item->id . dg::menuid($item->catid));
      }
    }
    $rows[] = $images;
    if (count($rows)) {
      foreach ($rows as $row) {
        $new_row = array();
        foreach ($row as $key => $title) {
          $new_row[] = $title;
        }
        $results = array_merge((array) $results, (array) $new_row);
      }
    }
    $this->_total = count($results);
    if ($this->getState('list.limit') > 0) {
      $results = array_splice($results, $this->getState('list.start'), $this->getState('list.limit'));
    } else {
      $results = $results;
    }
    return $results;
  }

  public function getTotal()
  {
    return $this->_total;
  }

  public function getPagination()
  {
    if (empty($this->_pagination)) {
      $this->_pagination = new JPagination($this->getTotal(), $this->getState('list.start'), $this->getState('list.limit'));
    }
    return $this->_pagination;
  }

  public function getCategory()
  {
    $app   = JFactory::getApplication();
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('id, title, description, metadesc, access');
    $query->from($db->qn('#__datsogallery_categories'));
    $query->where($db->qn('id') . ' = ' . $this->state->get('category.id'));
    if (JLanguageMultilang::isEnabled()) {
      $query->where($db->qn('language') . ' IN (' . $db->q(JFactory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');
    }
    $db->setQuery($query);
    $category = $db->loadObject();
    if (empty($category)) {
      $app->enqueueMessage(JText::_('COM_DATSOGALLERY_ERROR_PAGE_NOT_FOUND'), 'error');
      $app->redirect(JRoute::_('index.php?option=com_datsogallery&view=categories' . dg::itemId(), false), 301);
    } else {
      return $category;
    }
  }

  public function getPath()
  {
    $return    = array();
    $app = JFactory::getApplication();
    $menu = $app->getMenu()->getActive();
    $parent_id = $this->getState('category.id');
    if (!isset($menu->query['catid'])) {
      $row       = JTable::getInstance('Category', 'DatsogalleryTable');
      while ($parent_id > 1) {
        $row->load($parent_id);
        $parent_id  = $row->parent_id;
        $obj        = new stdClass();
        $obj->title = $row->title;
        $obj->link  = JRoute::_("index.php?option=com_datsogallery&amp;view=category&amp;catid=" . $row->id);
        $return[]   = $obj;
      }
      krsort($return);
    }
    return $return;
  }

  public function check_category()
  {
    $db           = JFactory::getDbo();
    $app          = JFactory::getApplication();
    $user         = JFactory::getUser();
    $settings     = dg::getConfig();
    $canCreateTop = $user->authorise('core.create.top', 'com_datsogallery');
    $and          = ($canCreateTop && !dg::is_root()) ? '' : $db->qn('created_by') . ' = ' . (int) $user->get('id');
    if ($settings->users_categories && (int) $user->get('id')) {
      $title       = $app->input->getString('title', '');
      $description = $app->input->getString('description', '');
      $language    = $app->input->getString('categorylanguage', '');
      $exists      = JString::strtolower(trim($title));
      $query       = $db->getQuery(true);
      $query->select('count(id)');
      $query->from($db->qn('#__datsogallery_categories'));
      $query->where($db->qn('created_by') . ' = ' . (int) $user->get('id'));
      $db->setQuery($query);
      $count = $db->loadResult();
      $query->clear();
      $query->select('title');
      $query->from($db->qn('#__datsogallery_categories'));
      $query->where($db->qn('title') . ' LIKE ' . $db->q($exists));
      $query->where($and);
      $db->setQuery($query);
      $name = $db->loadResult();
      if ($count >= ($settings->max_cats_per_member + 1) && !dg::is_root()) {
        echo 'maxcat';
        exit;
      } elseif (empty($exists)) {
        echo 'nocat';
        exit;
      } elseif (JString::strtolower(trim($name)) == $exists) {
        echo 'exist';
        exit;
      } else {
        $this->save_category($title, $description, $language);
        $this->refresh_categories($title);
        exit;
      }
    } else {
      exit;
    }
  }

  public function save_category($title, $description, $language = '*')
  {

    $app          = JFactory::getApplication();
    $access       = $app->get('access');
    $settings     = dg::getConfig();
    $user         = JFactory::getUser();
    $date         = JFactory::getDate();
    $isroot       = $user->authorise('core.admin');
    $canCreateTop = $user->authorise('core.create.top', 'com_datsogallery');
    $input        = $app->input;
    $post         = $input->post->getArray();
    $table        = $this->getTable();
    $data         = array();

    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('MIN(id)')
      ->from($db->qn('#__datsogallery_categories'))
      ->where($db->qn('created_by') . ' = ' . (int) $user->get('id'))
      ->where($db->qn('parent_id') . ' = ' . (int) $settings->members_category);
    $db->setQuery($query);
    $parent_id = $db->loadResult();

    $mc               = new stdClass();
    $mc->title        = $title;
    $mc->alias        = dg::sanitize($title);
    $mc->description  = $description;
    $mc->published    = 1;
    $mc->approved     = (($settings->users_categories && !$settings->category_approval) || $isroot) ? 1 : 0;
    $mc->access       = $access;
    $mc->created_by   = $user->id;
    $mc->created_time = $date->toSql();
    $mc->language     = (JLanguageMultilang::isEnabled() && !dg::falang_involved()) ? $language : '*';
    $mc->parent_id    = $parent_id ? $parent_id : (int) $settings->members_category;
    if (!$isroot && !$canCreateTop) {
      $mc->ordering = $table->getNextOrder('parent_id = ' . $parent_id);
    } else {
      $mc->ordering = $table->getNextOrder('parent_id = 1');
    }
    $result = $db->insertObject('#__datsogallery_categories', $mc);

    if ($result) {
      $new_id = $db->insertid();
      if (!$table->rebuild($table->id, $table->lft, $table->level, $table->path)) {
        $this->setError($table->getError());
        return false;
      }
    }
    if ($settings->category_notify_admin && !$isroot):
      $body = sprintf(JText::_('COM_DATSOGALLERY_COMMENT_NOTIFY_HELLO'), JText::_('JADMINISTRATOR')) . "\r\n\r\n";
      $body .= sprintf(JText::_('COM_DATSOGALLERY_CATEGORY_NOTIFY_ADMIN'), $user->get('name'), $title) . "\r\n\r\n";
      $body .= JText::_('COM_DATSOGALLERY_MAIL_MSG');
      $subject = sprintf(JText::_('COM_DATSOGALLERY_CATEGORY_NOTIFY_SUBJECT'), $app->getCfg('sitename'));
      dg::sendmail('', $subject, $body);
    endif;
    return true;
  }

  public function refresh_categories($title)
  {
    $user  = JFactory::getUser();
    $db    = JFactory::getDbo();
    $query = $db->getQuery(true);
    $query->select('id');
    $query->from('#__datsogallery_categories');
    $query->where('title = ' . $db->q($title));
    $query->where('created_by = ' . (int) $user->get('id'));
    $db->setQuery($query);
    $catid = $db->loadResult();
    $list  = dg::getOptions($catid, 'gcatid', 'id="gcatid" onchange="userCat(this.value);"');
    echo $list;
  }

  public function addMeta() {
    $limit = dg::getConfig()->meta_characters_limit;
    $app   = JFactory::getApplication();
    $pk    = $app->input->getInt('catid');
    $id    = isset($pk) ? (int) $pk : (int) $this->getState('category.id');
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('description, metadesc')
      ->from($db->qn('#__datsogallery_categories'))
      ->where($db->qn('id') . ' = ' . $id);
    $db->setQuery($query);
    $meta = $db->loadObject();
    if (!empty($meta->description) && empty($meta->metadesc)) {
      $description = JHtml::_('string.truncate', $meta->description, $limit, true, false);
      $query->clear()
        ->update($db->qn('#__datsogallery_categories'))
        ->set($db->qn('metadesc') . ' = ' . $db->q($description))
        ->where($db->qn('id') . ' = ' . $id);
      $db->setQuery($query)->execute();
    }
  }
}
