<?php
/**
 * @package    com_datsogallery
 * @author     Andrey Datso <support@datso.fr>
 * @copyright  (c) 2006 - 2020 Andrey Datso. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

use Joomla\Registry\Registry;
use Joomla\String\StringHelper;
use Joomla\Utilities\ArrayHelper;

jimport('joomla.application.component.modeladmin');

JLoader::register('dg', JPATH_ADMINISTRATOR . '/components/com_datsogallery/helpers/datsogallery.php');

class DatsogalleryModelImage extends JModelAdmin
{

  protected $text_prefix = 'COM_DATSOGALLERY';

  public function getTable($type = 'Image', $prefix = 'DatsogalleryTable', $config = array())
  {
    return JTable::getInstance($type, $prefix, $config);
  }

  public function batchCopy($value, $pks, $contexts)
  {
    $categoryId = (int) $value;
    $newIds     = array();
    while (!empty($pks)) {
      $pk = array_shift($pks);
      $this->table->reset();
      if (!$this->table->load($pk)) {
        if ($error = $this->table->getError()) {
          $this->setError($error);
          return false;
        } else {
          $this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_BATCH_MOVE_ROW_NOT_FOUND', $pk));
          continue;
        }
      }
      $this->table->id        = 0;
      $this->table->hits      = 0;
      $this->table->downloads = 0;
      $this->table->state     = 0;
      $this->table->catid     = $categoryId;
      $this->table->featured  = 0;
      if (!$this->table->check()) {
        $this->setError($this->table->getError());
        return false;
      }
      if (!$this->table->store()) {
        $this->setError($this->table->getError());
        return false;
      }
      $newId       = $this->table->get('id');
      $newIds[$pk] = $newId;
    }
    $this->cleanCache();
    return $newIds;
  }

  public function batchMove($value, $pks, $contexts)
  {
    $this->user = JFactory::getUser();
    $categoryId = (int) $value;
    foreach ($pks as $pk) {
      if (!$this->user->authorise('core.edit', $contexts[$pk])) {
        $this->setError(JText::_('JLIB_APPLICATION_ERROR_BATCH_CANNOT_EDIT'));
        return false;
      }
      if (!$this->table->load($pk)) {
        if ($error = $this->table->getError()) {
          $this->setError($error);
          return false;
        } else {
          $this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_BATCH_MOVE_ROW_NOT_FOUND', $pk));
          continue;
        }
      }
      $this->table->catid = $categoryId;
      if (!$this->table->check()) {
        $this->setError($this->table->getError());
        return false;
      }
      if (!$this->table->store()) {
        $this->setError($this->table->getError());
        return false;
      }
    }
    $this->cleanCache();
    return true;
  }

  public function getForm($data = array(), $loadData = true)
  {
    $app  = JFactory::getApplication();
    $form = $this->loadForm('com_datsogallery.image', 'image', array('control' => 'jform', 'load_data' => $loadData));
    if (empty($form)) {
      return false;
    }
    return $form;
  }

  protected function loadFormData()
  {
    $data = JFactory::getApplication()->getUserState('com_datsogallery.edit.image.data', array());
    if (empty($data)) {
      $data  = $this->getItem();
      $array = array();
      foreach ((array) $data->catid as $value):
        if (!is_array($value)):
          $array[] = $value;
        endif;
      endforeach;
      $data->catid = implode(',', $array);
    }
    return $data;
  }

  public function getItem($pk = null)
  {
    if ($item = parent::getItem($pk)) {
      $registry = new Registry($item->metadata);
      $item->metadata = $registry->toArray();
    }
    return $item;
  }

  protected function prepareTable($table)
  {
    jimport('joomla.filter.output');
    if (empty($table->id)) {
      if (@$table->ordering === '') {
        $db = JFactory::getDbo();
        $db->setQuery('SELECT MAX(ordering) FROM #__datsogallery_images');
        $max             = $db->loadResult();
        $table->ordering = $max + 1;
      }
    }
  }

  public function featured(&$pks, $value = 0)
  {
    $pks = (array) $pks;
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    $table = $this->getTable();
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_images');
      $query->set('featured = ' . (int) $value);
      $query->where('id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    $table->reorder();
    return true;
  }

  public function publish(&$pks, $value = 0)
  {
    $pks = (array) $pks;
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    $table = $this->getTable();
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_images');
      $query->set('state = ' . (int) $value);
      $query->where('id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    $table->reorder();
    return true;
  }

  public function approved(&$pks, $value = 0)
  {
    $pks = (array) $pks;
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    $table = $this->getTable();
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_images');
      $query->set('approved = ' . (int) $value);
      $query->where('id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    $table->reorder();
    return true;
  }

  public function review(&$pks, $value = 0)
  {
    $pks = (array) $pks;
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->delete('#__datsogallery_reports');
      $query->where('image_id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    return true;
  }

  public function publishcomment(&$pks, $value = 0)
  {
    $pks = (array) $pks;
    $settings = dg::getConfig();
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_comments');
      $query->set('state = ' . (int) $value);
      $query->where('id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    if ($settings->comments_approval && $value == 1) {
      foreach ($pks as $pk) {
         dg::comment_approval_notify($pk);
      }
    }
    return true;
  }

  public function getTags()
  {
    $id    = $this->getState('image.id');
    $db    = JFactory::getDbo();
    $query = $db->getQuery(true);
    $query->select('tag')
      ->from('#__datsogallery_tags')
      ->where('image_id = ' . $id);
    $db->setQuery($query);
    return $db->loadColumn();
  }

  public function addTags($tags)
  {
    $settings = dg::getConfig();
    $user = JFactory::getUser();
    $date = JFactory::getDate()->toSql();
    $table = $this->getTable();
    $nextid = $table->getNextId();
    $c = 0;
    foreach($tags as $k => $t) {
      $t = trim($t);
      if (JString::strlen($t) < $settings->min_tag_chars) {
        continue;
      } elseif (++$c > $settings->max_tags_per_image) {
        break;
      }
      $tag           = JTable::getInstance('Tag', 'DatsogalleryTable');
      $tag->image_id = $nextid;
      $tag->user_id  = $user->id;
      $tag->created  = $date;
      $tag->tag      = $t;
      $tag->state    = 1;
      if (!$tag->store()) {
        $this->setError($t->getError());
        return false;
      }
    }
    return true;
  }

  public function updateOriginal($id, $original)
  {
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->update($db->qn('#__datsogallery_images'))
      ->set($db->qn('original') . ' = ' . $db->q($original))
      ->where($db->qn('id') . ' = ' . $id);
    $db->setQuery($query);
    $db->execute();
    return true;
  }
}
