<?php
/**
 * @package    com_datsogallery
 * @author     Andrey Datso <support@datso.fr>
 * @copyright  (c) 2006 - 2020 Andrey Datso. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

use Joomla\Registry\Registry;
use Joomla\CMS\Session\Session;

class DatsogalleryModelImage extends JModelItem
{

  protected $_item    = null;
  protected $data     = null;
  protected $_context = 'com_datsogallery.image';

  public function __construct($config = array())
  {
    parent::__construct($config);
  }

  public function getTable($type = 'Image', $prefix = 'DatsogalleryTable', $config = array())
  {
    $this->addTablePath(JPATH_COMPONENT_ADMINISTRATOR . '/tables');
    return JTable::getInstance($type, $prefix, $config);
  }

  protected function populateState()
  {
    $app   = JFactory::getApplication();
    $user  = JFactory::getUser();
    $limit = $app->input->get('limit', 10, 'uint');
    $this->setState('list.limit', $limit);
    $limitstart = $app->input->get('limitstart', 0, 'uint');
    $this->setState('list.start', $limitstart);

    if ($app->input->get('layout') == 'edit') {
      $pk = $app->input->getInt('id');
      $app->setUserState('com_datsogallery.edit.image.id', $pk);
    } else {
      $pk = $app->input->getInt('id');
    }
    $settings = dg::getConfig();
    if ($settings->extract_color) {
      $obj = dg::imageobject($pk);
      if ($obj) {
        dg::savepalette($obj->original, $obj->id, $obj->catid);
      }
    }
    $this->setState('image.id', $pk);
    $params = $app->getParams();
    $this->setState('params', $params);
  }

  public function getItem($pk = null)
  {
    $app = JFactory::getApplication();
    $pk  = (!empty($pk)) ? $pk : (int) $this->getState('image.id');
    if (!isset($this->_item[$pk])) {
      try {
        $user     = JFactory::getUser();
        $groups   = implode(',', $user->getAuthorisedViewLevels());
        $db       = $this->getDbo();
        $nullDate = $db->q($db->getNullDate());
        $date     = JFactory::getDate();
        $nowDate  = $db->q($date->toSql());
        $query    = $db->getQuery(true);
        $query->select('a.*, a.title, c.access AS category_access');
        $query->from('#__datsogallery_images as a');
        $query->where($db->qn('a.access') . ' IN (' . $groups . ')');
        $query->where($db->qn('a.state') . ' = 1');
        $query->where($db->qn('a.approved') . ' = 1');
        $query->where('(' . $db->qn('a.publish_up') . ' = ' . $nullDate . ' OR ' . $db->qn('a.publish_up') . ' <= ' . $nowDate . ')');
        $query->where('(' . $db->qn('a.publish_down') . ' = ' . $nullDate . ' OR ' . $db->qn('a.publish_down') . ' >= ' . $nowDate . ')');
        $query->join('LEFT', '#__datsogallery_categories AS c ON c.id = a.catid');
        $query->where($db->qn('c.access') . ' IN (' . $groups . ')');
        $query->where($db->qn('c.published') . ' = 1');
        $query->where($db->qn('c.approved') . ' = 1');
        $query->where('(' . $db->qn('c.publish_up') . ' = ' . $nullDate . ' OR ' . $db->qn('c.publish_up') . ' <= ' . $nowDate . ')');
        $query->where('(' . $db->qn('c.publish_down') . ' = ' . $nullDate . ' OR ' . $db->qn('c.publish_down') . ' >= ' . $nowDate . ')');
        if (JLanguageMultilang::isEnabled()) {
          $query->where($db->qn('a.language') . ' IN (' . $db->q(JFactory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');
          $query->where($db->qn('c.language') . ' IN (' . $db->q(JFactory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');
        }
        $query->where('a.id = ' . (int) $pk);
        $db->setQuery($query);
        $data = $db->loadObject();
        if (empty($data)) {
          $app->enqueueMessage(JText::_('COM_DATSOGALLERY_ERROR_PAGE_NOT_FOUND'), 'error');
          $app->redirect(JRoute::_('index.php?option=com_datsogallery&view=categories' . dg::itemId(), false), 301);
        }
        $registry = new Registry($data->metadata);
        $data->metadata = $registry;
        $this->_item[$pk] = $data;
      } catch (Exception $e) {
        if ($e->getCode() == 404) {
          $app->enqueueMessage($e->getMessage(), 'error');
        } else {
          $this->setError($e);
          $this->_item[$pk] = false;
        }
      }
    }
    return $this->_item[$pk];
  }

  public function getPath()
  {
    $return    = array();
    $app = JFactory::getApplication();
    $menu = $app->getMenu()->getActive();
    $parent_id = $app->input->getInt('catid', 0);
    if (!isset($menu->query['catid'])) {
      $row       = JTable::getInstance('Category', 'DatsogalleryTable');
      while ($parent_id > 1) {
        $row->load($parent_id);
        $parent_id  = $row->parent_id;
        $obj        = new stdClass();
        $obj->title = $row->title;
        $obj->link  = JRoute::_("index.php?option=com_datsogallery&amp;view=category&amp;catid=" . $row->id);
        $return[]   = $obj;
      }
      krsort($return);
    }
    return $return;
  }

  public function checkin($id = null)
  {
    $id = (!empty($id)) ? $id : (int) $this->getState('image.id');
    if ($id) {
      $table = $this->getTable();
      if (method_exists($table, 'checkin')) {
        if (!$table->checkin($id)) {
          $this->setError($table->getError());
          return false;
        }
      }
    }
    return true;
  }

  public function checkout($id = null)
  {
    $id = (!empty($id)) ? $id : (int) $this->getState('image.id');
    if ($id) {
      $table = $this->getTable();
      $user  = JFactory::getUser();
      if (method_exists($table, 'checkout')) {
        if (!$table->checkout($user->get('id'), $id)) {
          $this->setError($table->getError());
          return false;
        }
      }
    }
    return true;
  }

  public function save($data)
  {
    $app   = JFactory::getApplication();
    $id    = (!empty($data['id'])) ? $data['id'] : (int) $this->getState('image.id');
    $state = (!empty($data['state'])) ? 1 : 0;
    $user  = JFactory::getUser();
    if ($id) {
      $authorised    = $user->authorise('core.edit', 'com_datsogallery.image.' . $id)
      || $authorised = $user->authorise('core.edit.own', 'com_datsogallery.image.' . $id);
      if ($user->authorise('core.edit.state', 'com_datsogallery.image.' . $id) !== true && $state == 1) {
        $data['state'] = 0;
      }
    } else {
      $authorised = $user->authorise('core.create', 'com_datsogallery');
      if ($user->authorise('core.edit.state', 'com_datsogallery.image.' . $id) !== true && $state == 1) {
        $data['state'] = 0;
      }
    }
    if ($authorised !== true) {
      $this->setError(JText::_('JERROR_ALERTNOAUTHOR'));
      return false;
    }
    $table = $this->getTable();
    if ($table->save($data) === true) {
      return $id;
    } else {
      return false;
    }
  }

  public function getImage($id)
  {
    static $cache = array();
    if (!isset($cache[$id])) {
      $table = $this->getTable();
      if ($table->load($id)) {
        $cache[$id] = $table;
      } else {
        $cache[$id] = false;
      }
    }
    return $cache[$id];
  }

  public function delete($id)
  {
    $user          = JFactory::getUser();
    $authorised    = $user->authorise('core.delete', 'com_datsogallery.image.' . $id)
    || $authorised = $user->authorise('core.delete.own', 'com_datsogallery.image.' . $id);
    if ($authorised !== true) {
      $this->setError(JText::_('JERROR_ALERTNOAUTHOR'));
      return false;
    }
    if ($image = $this->getImage($id)) {
      return $image->delete($id);
    }
    return false;
  }

  public function hit($pk = 0)
  {
    $settings = dg::getConfig();
    $user_ip  = dg::client_ip();
    $input    = JFactory::getApplication()->input;
    $pk       = (!empty($pk)) ? $pk : (int) $this->getState('image.id');
    if ($settings->hits_control && !$this->is_viewed($pk) || (!$settings->hits_control)) {
      $table = JTable::getInstance('Image', 'DatsogalleryTable');
      $table->load($pk);
      $table->hit($pk);
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->insert('#__datsogallery_hits')
        ->set($db->qn('id') . ' = ' . $db->q(''))
        ->set($db->qn('image_id') . ' = ' . $db->q($pk))
        ->set($db->qn('user_ip') . ' = ' . $db->q($user_ip));
      $db->setQuery($query);
      $db->execute();
    }
    return true;
  }

  public function is_viewed($id)
  {
    $user    = JFactory::getUser();
    $user_ip = dg::client_ip();
    $db      = $this->getDbo();
    $query   = $db->getQuery(true);
    $query->select('id')
      ->from('#__datsogallery_hits')
      ->where('user_ip = ' . $db->q($user_ip))
      ->where('image_id = ' . $id);
    $db->setQuery($query);
    return $db->loadResult();
  }

  public function getCategory()
  {
    $image_id = (int) $this->getState('image.id');
    $db       = $this->getDbo();
    $query    = $db->getQuery(true);
    $query->select('c.title')
      ->from('#__datsogallery_categories AS c')
      ->join('LEFT', '#__datsogallery_images AS a ON a.catid = c.id')
      ->where('a.id = ' . $image_id);
    $db->setQuery($query);
    return $db->loadObject();
  }

  public function getTags()
  {
    $image_id = (int) $this->getState('image.id');
    $lang     = JFactory::getLanguage()->getTag();
    $db       = $this->getDbo();
    $query    = $db->getQuery(true);
    $query->select('id, tag')
      ->from('#__datsogallery_tags')
      ->where('image_id = ' . $image_id)
      ->where($db->qn('language') . ' IN (' . $db->q($lang) . ',' . $db->q('*') . ')');
    $db->setQuery($query);
    return $db->loadObjectList();
  }

  public function getColors()
  {
    $image_id = (int) $this->getState('image.id');
    $db       = $this->getDbo();
    $query    = $db->getQuery(true);
    $query->select(array('hex'))
      ->from('#__datsogallery_colors')
      ->where('image_id = ' . $image_id);
    $db->setQuery($query);
    return $db->loadColumn();
  }

  public function getRelated()
  {
    $limit    = dg::getConfig('image_related_limit');
    $image_id = (int) $this->getState('image.id');
    $user     = JFactory::getUser();
    $groups   = implode(',', $user->getAuthorisedViewLevels());
    $tags     = array();
    $likes    = array();
    $results  = array();
    $tags     = $this->getTags();
    if (count($tags)) {
      foreach ($tags as $tag) {
        $likes[] = $tag->tag;
      }
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->select(array('distinct a.*', 'c.title AS category_title'));
      $query->from($db->qn('#__datsogallery_images', 'a'));
      $query->join('INNER', $db->qn('#__datsogallery_categories', 'c') . ' ON (' . $db->qn('a.catid') . ' = ' . $db->qn('c.id') . ')');
      $query->join('INNER', $db->qn('#__datsogallery_tags', 't') . ' ON (' . $db->qn('a.id') . ' = ' . $db->qn('t.image_id') . ')');
      $query->where('(' . $db->qn('t.tag') . ' LIKE "' . implode('" OR ' . $db->qn('t.tag') . ' LIKE "', $likes) . '")');
      $query->where('NOT ' . $db->qn('a.id') . ' = ' . $image_id);
      $query->where($db->qn('a.access') . ' IN (' . $groups . ')');
      $query->where($db->qn('c.access') . ' IN (' . $groups . ')');
      $query->order($query->Rand());
      $query->setLimit($limit);
      $db->setQuery($query);
      //echo ($query->__toString());
      try {
        $results = $db->loadObjectList();
      } catch (RuntimeException $e) {
        JFactory::getApplication()->enqueueMessage(JText::_('JERROR_AN_ERROR_HAS_OCCURRED'), 'error');
      }
    }
    return $results;
  }

  public function getOwner()
  {
    $image_id = (int) $this->getState('image.id');
    $db       = $this->getDbo();
    $query    = $db->getQuery(true);
    $query->select('u.name')
      ->from('#__users AS u')
      ->join('LEFT', '#__datsogallery_images AS a ON a.created_by = u.id')
      ->where('a.id = ' . $image_id);
    $db->setQuery($query);
    return $db->loadObject();
  }

  public function sendReport()
  {
    $user_ip  = dg::client_ip();
    $user     = JFactory::getUser();
    $settings = dg::getConfig();
    $input    = JFactory::getApplication()->input;
    $title    = $input->getString('title', '');
    $message  = $input->getString('message', '');
    $image_id = $input->getInt('image_id', 0);
    $type     = $input->getString('type', '');
    $date     = JFactory::getDate();
    $db       = $this->getDbo();
    $query    = $db->getQuery(true);
    $columns  = array('id', 'title', 'message', 'image_id', 'type', 'created_by', 'ip', 'created', 'state');
    $values   = array($db->q(''), $db->q($title), $db->q($message), $image_id, $db->q($type), $user->id, $db->q($user_ip), $db->q($date->toSql()), 1);
    $query->insert($db->qn('#__datsogallery_reports'))->columns($db->qn($columns))->values(implode(',', $values));
    $db->setQuery($query);
    $db->execute();
    $query->clear();
    $query->select('title, original, catid')->from('#__datsogallery_images')->where('id = ' . $image_id);
    $db->setQuery($query);
    $obj = $db->loadObject();
    //$cacheimage = getCacheFile($obj->original, $settings->image_width_details, $settings->image_height_details, $obj->catid, $settings->image_aspect_ratio_details);
    //$imagemime = getimagesize($cacheimage);
    $mailer = JFactory::getMailer();
    $config = JFactory::getConfig();
    $sender = array(
      $config->get('mailfrom'),
      $config->get('fromname')
    );
    $mailer->setSender($sender);
    $mailer->addRecipient($config->get('mailfrom'));
    $body = '<html>'
    . '<head>'
    . '  <meta name="viewport" content="width=device-width">'
    . '  <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">'
    . '  <title>' . JText::_('COM_DATSOGALLERY_REPORT_IMAGE_MAIL_TITLE') . '</title>'
    . '</head>'
    . '<body style="background: #F5F5F5;-webkit-font-smoothing: antialiased;height: 100%;-webkit-text-size-adjust: none;margin: 10px 0;width: 100%;">'
    . '<div style="max-width:480px;margin:0 auto;border:1px solid #eee;padding:20px 40px;background:#FCFCFC;font-family: \'Lucida Sans Unicode\';font-size:14px">'
    . '    <h3 style="border-bottom: 1px solid #eee;line-height: 40px;text-align: center;font-size: 22px;color: #111111;font-family: \'Lucida Sans Unicode\';font-weight: 200;margin: 10px 0 10px;">' . $config->get('sitename') . '</h3>'
    . '    <p style="font-size: 14px;font-weight: normal;margin-bottom: 10px;">' . JText::sprintf('COM_DATSOGALLERY_REPORT_IMAGE_MAIL_USER', $title) . '</p>'
    . '    <div style="border: 1px solid #eee;border-left-width: 3px;padding: 10px;background: #fff;"><em>' . nl2br($message . '<p>IP: ' . $user_ip . '</p>') . '</em></div>'
    . '    <h3 style="text-align: center;font-size: 22px;color: #111111;font-family: \'Lucida Sans Unicode\';font-weight: 200;line-height: 1.2em;margin: 10px 0 10px;">' . $obj->title . '</h3>'
    . '    <p style="font-size: 14px;font-weight: normal;margin-bottom: 10px;">' . JText::_('COM_DATSOGALLERY_REPORT_IMAGE_MAIL_LOGIN') . '</p>'
    . '    <p style="border-top: 1px solid #eee;padding-top: 15px;font-size: 12px;font-weight: normal;margin-bottom: 10px;"><em>' . JText::_('COM_DATSOGALLERY_MAIL_NOREPLY') . '</em></p>'
      . '</div>'
      . '</body>'
      . '</html>';
    $mailer->isHTML(true);
    $mailer->setSubject(JText::_('COM_DATSOGALLERY_REPORT_IMAGE_MAIL_TITLE'));
    $mailer->setBody($body);
    $mailer->Send();
    return true;
  }

  public function updateOriginal($id, $original)
  {
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->update($db->qn('#__datsogallery_images'))
      ->set($db->qn('original') . ' = ' . $db->q($original))
      ->where($db->qn('id') . ' = ' . $id);
    $db->setQuery($query);
    $db->execute();
    return true;
  }

  public function addtofavorite($id)
  {
    $user  = JFactory::getUser();
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('id')
      ->from($db->qn('#__datsogallery_favorites'))
      ->where($db->qn('image_id') . ' = ' . (int) $id)
      ->where($db->qn('user_id') . ' = ' . $user->id);
    $db->setQuery($query);
    if (!$db->loadResult()) {
      $columns = array('image_id', 'user_id');
      $values  = array($id, $user->id);
      $query->clear()
        ->insert($db->qn('#__datsogallery_favorites'))
        ->columns($db->qn($columns))
        ->values(implode(',', $values));
      $db->setQuery($query);
      $array = array('status' => 'favorited', 'msg' => JText::_('COM_DATSOGALLERY_TOGGLE_FAVORITED'));
    } else {
      $query->clear()
        ->delete($db->qn('#__datsogallery_favorites'))
        ->where($db->qn('image_id') . ' = ' . (int) $id)
        ->where($db->qn('user_id') . ' = ' . $user->id);
      $db->setQuery($query);
      $array = array('status' => 'unfavorited', 'msg' => JText::_('COM_DATSOGALLERY_TOGGLE_UNFAVORITED'));
    }
    $db->execute();
    return json_encode($array);
  }

  public function updateLikes()
  {
    $user    = JFactory::getUser();
    $user_ip = dg::client_ip();
    $input   = JFactory::getApplication()->input;
    $act     = $input->getCmd('act');
    $id      = $input->getInt('id');
    $db      = $this->getDbo();
    if ($act == 'like') {
      $query = $db->getQuery(true);
      if (!$this->getLikes($id)) {
        $columns = array('image_id', 'user_id', 'ip', 'rate');
        $values  = array($id, $user->id, $db->q($user_ip), 1);
        $query->insert($db->qn('#__datsogallery_likes'))
          ->columns($db->qn($columns))
          ->values(implode(',', $values));
        $db->setQuery($query);
        try {
          $db->execute();
        } catch (Exception $e) {
          return false;
        }
        $array = array('status' => 'liked', 'msg' => JText::_('COM_DATSOGALLERY_LIKE_COUNTED'), 'likes' => $this->getLikecount($id));
      } else {
        $query->clear();
        $query->delete($db->qn('#__datsogallery_likes'));
        if ($user->id) {
          $query->where('(user_id = ' . $user->id . ' or ip = ' . $db->q($user_ip) . ')');
        } else {
          $query->where('ip = ' . $db->q($user_ip));
        }
        $query->where($db->qn('image_id') . ' = ' . $id);
        $db->setQuery($query);
        try {
          $db->execute();
        } catch (Exception $e) {
          return false;
        }
        $array = array('status' => 'disliked', 'msg' => JText::_('COM_DATSOGALLERY_LIKE_SUBTRACTED'), 'likes' => $this->getLikecount($id));
      }
    }
    return json_encode($array);
  }

  public function likeComment($id = null)
  {
    $user    = JFactory::getUser();
    $user_ip = dg::client_ip();
    $db      = $this->getDbo();
    $query   = $db->getQuery(true);
    if (!dg::is_likedcomment($id)) {
      $columns = array('comment_id', 'user_id', 'ip', 'like');
      $values  = array($id, $user->id, $db->q($user_ip), 1);
      $query->insert($db->qn('#__datsogallery_comment_likes'))
        ->columns($db->qn($columns))
        ->values(implode(',', $values));
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $e) {
        return false;
      }
      $array = array('status' => 'liked', 'msg' => JText::_('COM_DATSOGALLERY_LIKE_COUNTED'), 'likes' => dg::commentlikes($id));
    } else {
      $query->clear();
      $query->delete($db->qn('#__datsogallery_comment_likes'));
      if ($user->id) {
        $query->where('(user_id = ' . $user->id . ' or ip = ' . $db->q($user_ip) . ')');
      } else {
        $query->where('ip = ' . $db->q($user_ip));
      }
      $query->where($db->qn('comment_id') . ' = ' . $id);
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $e) {
        return false;
      }
      $array = array('status' => 'disliked', 'msg' => JText::_('COM_DATSOGALLERY_LIKE_SUBTRACTED'), 'likes' => dg::commentlikes($id));
    }
    return json_encode($array);
  }

  public function getLikes($image_id = null)
  {
    $user    = JFactory::getUser();
    $user_ip = dg::client_ip();
    $id      = $image_id ? $image_id : $this->getState('image.id');
    $db      = $this->getDbo();
    $query   = $db->getQuery(true);
    $query->select('count(*)');
    $query->from('#__datsogallery_likes');
    if ($user->id):
      $query->where('(user_id = ' . $user->id . ' or ip = ' . $db->q($user_ip) . ')');
    else:
      $query->where('ip = ' . $db->q($user_ip));
    endif;
    $query->where('rate = 1');
    $query->where('image_id = ' . $id);
    $db->setQuery($query);
    return $db->loadResult();
  }

  public function getRateall($image_id = null)
  {
    $id    = $image_id ? $image_id : $this->getState('image.id');
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('count(*)')
      ->from('#__datsogallery_likes')
      ->where('image_id = ' . $id);
    $db->setQuery($query);
    return $db->loadResult();
  }

  public function getLikecount($image_id = null)
  {
    $id    = (int) $image_id ? (int) $image_id : $this->getState('image.id');
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('count(*)')
      ->from('#__datsogallery_likes')
      ->where('rate = 1')
      ->where('image_id = ' . (int) $id);
    $db->setQuery($query);
    return $db->loadResult();
  }

  public function getComments()
  {
    $id    = (int) $this->getState('image.id');
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select($this->getState('list.select', '*'))
      ->from('#__datsogallery_comments')
      ->where('image_id = ' . $id)
      ->where('state = 1')
      ->where('parent_id = 0')
      ->order('id DESC');
    $db->setQuery($query);
    $results      = $db->loadObjectList();
    $this->_total = count($results);
    if ($this->getState('list.limit') > 0) {
      $results = array_splice($results, $this->getState('list.start'), $this->getState('list.limit'));
    } else {
      $results = $results;
    }
    return $results;
  }

  public function getTotal()
  {
    return $this->_total;
  }

  public function getPagination()
  {
    if (empty($this->_pagination)) {
      $this->_pagination = new JPagination($this->getTotal(), $this->getState('list.start'), $this->getState('list.limit'));
    }
    return $this->_pagination;
  }

  public function getParentcomments()
  {
    $id    = (int) $this->getState('image.id');
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('*')
      ->from('#__datsogallery_comments')
      ->where('image_id = ' . $id)
      ->where('state = 1')
      ->where('parent_id != 0')
      ->order('id DESC');
    $db->setQuery($query);
    $comments = $db->loadObjectList();
    return $comments;
  }

  public function updateComment($id, $text)
  {
    $app   = JFactory::getApplication();
    $user  = JFactory::getUser();
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->update($db->qn('#__datsogallery_comments'))
      ->set($db->qn('text') . ' = ' . $db->q($text))
      ->where($db->qn('id') . ' = ' . (int) $id);
    $db->setQuery($query);
    $db->execute();
    $query->clear()
      ->select('text')
      ->from($db->qn('#__datsogallery_comments'))
      ->where($db->qn('id') . ' = ' . (int) $id);
    $db->setQuery($query);
    return $db->loadResult();
  }

  public function commentReply($id, $parent_id, $text, $link)
  {
    $user     = JFactory::getUser();
    $isadmin  = $user->authorise('core.admin');
    $settings = dg::getConfig();
    $ip       = dg::getIpAddress();
    if ($settings->comments_captcha && $settings->comments_captcha_type == 0 && $settings->comments_dnsbl && $this->is_blacklisted($ip)) {
      $array = array(
        'status' => 'blacklisted',
        'user_name' => JText::_('COM_DATSOGALLERY_SYSTEM'),
        'avatar' => '<span class="dg-system-avatar dg-error">S</span>',
        'text' => '<h4>' . JText::_("ERROR") . '</h4><p>'.JText::sprintf("COM_DATSOGALLERY_COMMENTS_DNSBL_NOTICE", '<a rel="nofollow" href="https://en.wikipedia.org/wiki/DNSBL" target="_blank">DNSBL</a>') . '</p>',
      );
    } else if ($settings->comments_captcha && !$user->id && (in_array($settings->comments_captcha_type, array(1, 2))) && $response !== true) {
      $body_text = 'COM_DATSOGALLERY_RECAPTCHA_' . str_replace('-', '_', str_replace('"', '', $response));
      $array     = array(
        'status' => 'recaptcha',
        'user_name' => JText::_('COM_DATSOGALLERY_SYSTEM'),
        'avatar' => '<span class="dg-system-avatar dg-error">S</span>',
        'text' => '<h4>' . JText::_("ERROR") . '</h4><p>'.JText::_(strtoupper($body_text)).'</p>',
      );
    } else
    if (dg::canreply((int) $id)) {
      if ($user->id) {
        $date              = JFactory::getDate();
        $table             = JTable::getInstance('Comment', 'DatsogalleryTable');
        $table->image_id   = (int) $id;
        $table->user_ip    = $ip;
        $table->user_name  = $user->name;
        $table->user_email = $user->email;
        $table->user_url   = '';
        $table->text       = $text;
        $table->date       = $date->toSql();
        $table->parent_id  = (int) $parent_id;
        $table->state      = ($settings->comments_approval && !$isadmin) ? 0 : 1;
        $table->created_by = $user->id;
        if (!$table->store()) {
          $this->setError($table->getError());
          return false;
        }
        /*if (!$settings->comments_approval) {
          dg::comment_notify($id, $user_name, $user_email, $text);
        }*/
        $array = array(
          'status' => ($settings->comments_approval && !$isadmin) ? 'approval' : 'success',
          'id' => $table->id,
          'avatar' => ($settings->comments_approval && !$isadmin) ? '<span class="dg-system-avatar dg-warning">S</span>' : dg::avatar($table->created_by, $table->id, false, true, false, 'comments'),
          'image_id' => $table->image_id,
          'user_name' => ($settings->comments_approval && !$isadmin) ? JText::_('COM_DATSOGALLERY_SYSTEM') : $table->user_name,
          'text' => ($settings->comments_approval && !$isadmin) ? JText::_('COM_DATSOGALLERY_COMMENTS_APPROVAL_MSG') : $table->text,
          'parent_id' => $table->parent_id,
          'state' => $table->state,
          'created_by' => $table->created_by,
          'reply_to' => dg::username($table->parent_id),
          'total' => dg::numcomments((int) $table->image_id)
        );
      } else {
        $array = array(
          'status' => 'unauthorized',
          'avatar' => '<span class="dg-system-avatar dg-warning">S</span>',
          'text' => JText::sprintf('COM_DATSOGALLERY_LEAVE_COMMENT_REPLY_MSG', '<a href="' . JRoute::_('index.php?option=com_users&amp;view=login&amp;return=' . $link) . '">' . JText::_('COM_DATSOGALLERY_LOGIN') . '</a>'),
        );
      }
    } else {
      $array = array(
        'status' => 'self',
        'avatar' => '<span class="dg-system-avatar dg-warning">S</span>',
        'text' => JText::_('COM_DATSOGALLERY_CANT_REPLY_YOURSELF'),
      );
    }
    return json_encode($array);
  }

  public function getCommentTime($id)
  {
    $app      = JFactory::getApplication();
    $settings = dg::getConfig();
    $table    = JTable::getInstance('Comment', 'DatsogalleryTable');
    $table->load($id);
    $date = dg::timelapse($table->date, $settings->comments_date_format);
    return $date;
  }

  public function getComment($id, $user_name, $user_email, $text)
  {
    $app      = JFactory::getApplication();
    $user     = JFactory::getUser();
    $isadmin  = $user->authorise('core.admin');
    $date     = JFactory::getDate();
    $settings = dg::getConfig();
    $ip       = dg::getIpAddress();
    $array    = array();
    $columns  = array($id, $user_name, $user_email, $text, $ip);
    $response = dg::recaptcha_response();
    if ($settings->comments_captcha && $settings->comments_captcha_type == 0 && $settings->comments_dnsbl && $this->is_blacklisted($ip)) {
      $array = array(
        'status' => 'blacklisted',
        'user_name' => JText::_('COM_DATSOGALLERY_SYSTEM'),
        'avatar' => '<span class="dg-system-avatar dg-error">S</span>',
        'text' => '<h4>' . JText::_("ERROR") . '</h4><p>'.JText::sprintf("COM_DATSOGALLERY_COMMENTS_DNSBL_NOTICE", '<a rel="nofollow" href="https://en.wikipedia.org/wiki/DNSBL" target="_blank">DNSBL</a>') . '</p>',
      );
    } else if ($settings->comments_captcha && !$user->id && (in_array($settings->comments_captcha_type, array(1, 2))) && $response !== true) {
      $body_text = 'COM_DATSOGALLERY_RECAPTCHA_' . str_replace('-', '_', str_replace('"', '', $response));
      $array     = array(
        'status' => 'recaptcha',
        'user_name' => JText::_('COM_DATSOGALLERY_SYSTEM'),
        'avatar' => '<span class="dg-system-avatar dg-error">S</span>',
        'text' => '<h4>' . JText::_("ERROR") . '</h4><p>'.JText::_(strtoupper($body_text)).'</p>',
      );
    } else {
      if (($id && $user->id && trim($text) != '') || (!$user->id && count(array_filter($columns)) == count($columns))) {
        $table             = JTable::getInstance('Comment', 'DatsogalleryTable');
        $table->image_id   = (int) $id;
        $table->user_ip    = $ip;
        $table->user_name  = $user_name;
        $table->user_email = $user_email;
        $table->user_url   = '';
        $table->text       = $text;
        $table->date       = $date->toSql();
        $table->parent_id  = 0;
        $table->state      = ($settings->comments_approval && !$isadmin) ? 0 : 1;
        $table->created_by = $user->id;
        if (!$table->store()) {
          $this->setError($table->getError());
          return false;
        }
        if (!$settings->comments_approval) {
          dg::comment_notify($id, $user_name, $user_email, $text);
        }
        $array = array(
          'status' => ($settings->comments_approval && !$isadmin) ? 'approval' : 'success',
          'id' => $table->id,
          'avatar' => ($settings->comments_approval && !$isadmin) ? '<span class="dg-system-avatar dg-warning">S</span>' : dg::avatar($table->created_by, $table->id, false, true, false, 'comments'),
          'image_id' => $table->image_id,
          'user_name' => ($settings->comments_approval && !$isadmin) ? JText::_('COM_DATSOGALLERY_SYSTEM') : $table->user_name,
          'text' => ($settings->comments_approval && !$isadmin) ? JText::_('COM_DATSOGALLERY_COMMENTS_APPROVAL_MSG') : $table->text,
          'parent_id' => $table->parent_id,
          'state' => $table->state,
          'created_by' => $table->created_by,
          'total' => dg::numcomments((int) $table->image_id)
        );
      } else {
        $array = array(
          'status' => 'unknown',
          'user_name' => JText::_('COM_DATSOGALLERY_SYSTEM'),
          'avatar' => '<span class="dg-system-avatar dg-error">S</span>',
          'text' => '<h4>' . JText::_("ERROR") . '</h4><p>' . JText::_("Something went wrong") . '</p>',
        );
      }
    }

    return json_encode($array);
  }

  public function deletecomment($id, $image_id)
  {
    if (dg::candeletecomment($id)) {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->delete($db->qn('#__datsogallery_comments'))
        ->where($db->qn('id') . ' = ' . (int) $id);
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $e) {
        return $e->getMessage();
      }
      return dg::numcomments((int) $image_id);
    }
  }

  public function getBlocked()
  {
    return dg::is_blocked();
  }

  public function is_blacklisted($ip)
  {
    $settings    = dg::getConfig();
    $excluded_ip = trim($settings->comments_dnsbl_excluded) ? explode(',', $settings->comments_dnsbl_excluded) : array();
    $listed      = $settings->comments_dnsbl_ratio ? $this->blacklist($ip, $settings->comments_dnsbl_percentage, 0) : in_array(1, $this->blacklist($ip));
    if (!in_array($ip, $excluded_ip) && $listed) {
      return true;
    } else {
      return false;
    }
  }

  public function blacklist($ip, $p = 10, $array = 1)
  {
    $settings      = dg::getConfig();
    $dnsbl_servers = explode(',', $settings->comments_dnsbl_servers);
    $lookups       = count($dnsbl_servers);
    $total         = 0;
    if ($ip) {
      $reverse_ip = implode(".", array_reverse(explode(".", $ip)));
      foreach ($dnsbl_servers as $host) {
        if (checkdnsrr($reverse_ip . "." . $host . ".", "A")) {
          $result["$host"] = 1;
          $total++;
        } else {
          $result["$host"] = 0;
        }
      }
    }
    $percent = ($total / $lookups) * 100;
    if ($array) {
      return $result;
    } else {
      return ($percent >= $p) ? 1 : 0;
    }
  }

  public function validateMail()
  {
    $input = JFactory::getApplication()->input;
    $email = $input->getString('user_email', '');
    if (function_exists('filter_var')) {
      if (filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
        return false;
      }
    }
    $domain = explode("@", $email);
    if (checkdnsrr(end($domain)) === false) {
      echo 'false';
    } else {
      echo 'true';
    }
  }

  public function validaterecaptcha()
  {
    return dg::recaptcha_response();
  }

  public function dgCaptcha()
  {
    $app     = JFactory::getApplication();
    $session = JFactory::getSession();
    $session->clear('dg-captcha');
    require_once JPATH_COMPONENT . '/helpers/captcha.php';
    $captcha = new captcha();
    $captcha->get('dg-captcha');
    $app->close();
  }

  public function validateCaptcha()
  {
    $input        = JFactory::getApplication()->input;
    $session      = JFactory::getSession();
    $dgcaptcha    = $session->get('dg-captcha');
    $dgcaptchaval = $input->get('dgcaptchaval', '', 'string');
    if (!empty($dgcaptcha) && strtolower($dgcaptchaval) != $dgcaptcha) {
      echo 'false';
    } else {
      echo 'true';
    }
  }

  public function dgCaptchar()
  {
    $app     = JFactory::getApplication();
    $session = JFactory::getSession();
    $session->clear('dgr-captcha');
    require_once JPATH_COMPONENT . '/helpers/captcha.php';
    $captcha = new captcha();
    $captcha->get('dgr-captcha');
    $app->close();
  }

  public function validateCaptchar()
  {
    $input        = JFactory::getApplication()->input;
    $session      = JFactory::getSession();
    $dgcaptcha    = $session->get('dgr-captcha');
    $dgcaptchaval = $input->get('dgrcaptchaval', '', 'string');
    if (!empty($dgcaptcha) && strtolower($dgcaptchaval) != $dgcaptcha) {
      echo 'false';
    } else {
      echo 'true';
    }
  }

  public function download($id)
  {
    $db    = $this->getDbo();
    $user  = JFactory::getUser();
    $query = $db->getQuery(true);
    $query->select('title, original')
      ->from($db->qn('#__datsogallery_images'))
      ->where(($db->qn('id') . ' = ' . (int) $id))
      ->where($db->qn('state') . ' = 1');
    $db->setQuery($query);
    return $db->loadObject();
  }

  public function downloadRecord($id)
  {
    $settings = dg::getConfig();
    $user_ip  = dg::client_ip();
    if ($settings->allow_downloads) {
      $db = $this->getDbo();
      $q  = $db->getQuery(true);
      $q->update($db->qn('#__datsogallery_images'));
      $q->set($db->qn('downloads') . ' = downloads + 1');
      $q->where($db->qn('id') . ' = ' . (int) $id);
      $db->setQuery($q)->execute();
      $q->clear();
      $q->insert('#__datsogallery_downloads');
      $q->set($db->qn('image_id') . ' = ' . (int) $id);
      $q->set($db->qn('user_ip') . ' = ' . $db->q($user_ip));
      $db->setQuery($q)->execute();
    }
    return true;
  }

  public function addTags($tags)
  {
    $settings = dg::getConfig();
    $user = JFactory::getUser();
    $date = JFactory::getDate()->toSql();
    $image = JTable::getInstance('Image', 'DatsogalleryTable');
    $nextid = $image->getNextId();
    $c = 0;
    foreach($tags as $k => $t) {
      $t = trim($t);
      if (JString::strlen($t) < $settings->min_tag_chars) {
        continue;
      } elseif (++$c > $settings->max_tags_per_image) {
        break;
      }
      $tag           = JTable::getInstance('Tag', 'DatsogalleryTable');
      $tag->image_id = $nextid;
      $tag->user_id  = $user->id;
      $tag->created  = $date;
      $tag->tag      = $t;
      $tag->state    = 1;
      if (!$tag->store()) {
        $this->setError($t->getError());
        return false;
      }
    }
    return true;
  }

  public function addMeta() {
    $limit = dg::getConfig()->meta_characters_limit;
    $app   = JFactory::getApplication();
    $pk    = $app->input->getInt('id');
    $id    = isset($pk) ? (int) $pk : (int) $this->getState('image.id');
    $db    = $this->getDbo();
    $query = $db->getQuery(true);
    $query->select('description, metadesc')
      ->from($db->qn('#__datsogallery_images'))
      ->where($db->qn('id') . ' = ' . $id);
    $db->setQuery($query);
    $meta = $db->loadObject();
    if (!empty($meta->description) && empty($meta->metadesc)) {
      $description = JHtml::_('string.truncate', $meta->description, $limit, true, false);
      $query->clear()
        ->update($db->qn('#__datsogallery_images'))
        ->set($db->qn('metadesc') . ' = ' . $db->q($description))
        ->where($db->qn('id') . ' = ' . $id);
      $db->setQuery($query)->execute();
    }
  }
}
