<?php
/**
 * @package    com_datsogallery
 * @author     Andrey Datso <support@datso.fr>
 * @copyright  (c) 2006 - 2020 Andrey Datso. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;
jimport('joomla.filesystem.folder');
jimport('joomla.filesystem.file');
require_once JPATH_ADMINISTRATOR . '/components/com_datsogallery/helpers/datsogallery.php';

class RealTimeUpload
{

  private $UPLOAD_SETTINGS = array(
    'uploadFolder' => '',
    'maxFileSize' => 120000,
    'extension' => ['png', 'jpg', 'jpeg', 'gif'],
    'rename' => false,
    'overwrite' => true,
    'maxFolderFiles' => 0,
    'maxFolderSize' => 0,
    'protectUploadDirectory' => false,
    'denyUploadDirectoryAccess' => false,
    'uniqueFilename' => false,
    'changeFile' => false,
    'originalName' => '',
    'avatarFilename' => false,
    'resizeCrop' => false,
    'returnLocation' => false,
    'maxWidth' => 0,
    'maxHeight' => 0
  );

  private $UPLOAD_NAME;
  private $UPLOAD_TMPNAME;
  private $UPLOAD_TYPE;
  private $UPLOAD_ERROR;
  private $UPLOAD_SIZE;
  private $UPLOAD_TOTALSIZE = 0;

  public function name()
  {
    return $this->UPLOAD_NAME;
  }

  public function tmpname()
  {
    return $this->UPLOAD_TMPNAME;
  }

  public function type()
  {
    return $this->UPLOAD_TYPE;
  }

  public function error()
  {
    return $this->UPLOAD_ERROR;
  }

  public function size()
  {
    return $this->UPLOAD_SIZE;
  }

  public function totalsize()
  {
    return $this->UPLOAD_TOTALSIZE;
  }

  public function sanitize($folder)
  {
    if (isset($folder)) {
      return $this->sanitizeFilename($folder, '');
    } else {
      return false;
    }
  }

  public function removeTmp($folder, $time)
  {
    if (!isset($time)) {
      $time = 24;
    }
    if (!isset($folder)) {
      $folder = ini_get('upload_tmp_dir') ? ini_get('upload_tmp_dir') : sys_get_temp_dir();
    }
    if (substr(ini_get('session.save_path'), 0, 4) != '/tmp') {
      foreach (glob(rtrim(ini_get('session.save_path'), '/') . '/sess_*') as $filename) {
        if (filemtime($filename) + ini_get('session.gc_maxlifetime') < time()) {
          @unlink($filename);
        }
      }
    }
    if (isset($folder)) {
      $files   = glob($folder . "/*");
      $now     = time();
      $counter = 0;
      foreach ($files as $file) {
        if (is_file($file)) {
          if ($now - filemtime($file) >= 60 * 60 * $time && pathinfo($file, PATHINFO_EXTENSION) == 'tmp') {
            unlink($file);
            $counter++;
          }
        }
      }
      return $counter;
    } else {
      return false;
    }
  }

  public function init($settings)
  {
    foreach ($this->UPLOAD_SETTINGS as $key => $value) {
      if (array_key_exists($key, $settings)) {
        $this->UPLOAD_SETTINGS[$key] = $settings[$key];
      }
    }
    if (!JFolder::exists($this->UPLOAD_SETTINGS['uploadFolder'])
      || !is_writable($this->UPLOAD_SETTINGS['uploadFolder'])) {
      JFolder::create($this->UPLOAD_SETTINGS['uploadFolder']);
    }
    if ($this->UPLOAD_SETTINGS['protectUploadDirectory']) {
      $p = $this->UPLOAD_SETTINGS['uploadFolder'] . '/.htaccess';
      if (!JFile::exists($p)) {
        $htContent = "php_flag engine off\nOptions -Indexes";
        if ($this->UPLOAD_SETTINGS['denyUploadDirectoryAccess']) {
          $htContent .= "\ndeny from all";
        }
        $ht = fopen($p, 'c');
        if (flock($ht, LOCK_SH | LOCK_NB) && flock($ht, LOCK_EX)) {
          fwrite($ht, $htContent);
          fflush($ht);
          flock($ht, LOCK_UN);
        }
        fclose($ht);
      }
    }
    if (!empty($_FILES)) {
      $this->getFileStatus();
    }
  }

  private function getFileStatus()
  {
    foreach ($_FILES as $index => $tmpName) {
      try {
        if (is_array($_FILES[$index]['tmp_name'])) {
          for ($i = 0, $l = count($_FILES[$index]['tmp_name']); $i < $l; $i++) {
            $this->UPLOAD_NAME    = $_FILES[$index]['name'][$i];
            $this->UPLOAD_TMPNAME = $_FILES[$index]['tmp_name'][$i];
            $this->UPLOAD_TYPE    = $_FILES[$index]['type'][$i];
            $this->UPLOAD_ERROR   = $_FILES[$index]['error'][$i];
            $this->UPLOAD_SIZE    = $_FILES[$index]['size'][$i];
            $this->UPLOAD_TOTALSIZE += $_FILES[$index]['size'][$i];
            $this->uploadFile();
          }
        } else {
          $this->UPLOAD_NAME    = $_FILES[$index]['name'];
          $this->UPLOAD_TMPNAME = $_FILES[$index]['tmp_name'];
          $this->UPLOAD_TYPE    = $_FILES[$index]['type'];
          $this->UPLOAD_ERROR   = $_FILES[$index]['error'];
          $this->UPLOAD_SIZE    = $_FILES[$index]['size'];
          $this->UPLOAD_TOTALSIZE += $_FILES[$index]['size'];
          $this->uploadFile();
        }
      } catch (Exception $e) {
        $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_IMAGE_CORRUPT'));
      }
    }
  }

  private function uploadFile()
  {
    $uploaded     = false;
    $uploadedFile = '';
    $cfg    = dg::getConfig();
    $x = explode(',', $cfg->extensions);
    $a = 'image/'.implode(',image/', $x);
    $allowed_mime = explode(',', $a);
    $this->getUploadFolderSize();

    switch ($this->UPLOAD_ERROR) {
      case UPLOAD_ERR_OK:
        break;

      case UPLOAD_ERR_NO_FILE:
        $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_NO_FILE_SENT'));

      case UPLOAD_ERR_INI_SIZE:
      case UPLOAD_ERR_FORM_SIZE:
        $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_EXCEEDED_FILESIZE_LIMIT'));
    }
    if (isset($_SERVER['HTTP_X_FILENAME']) && $_SERVER['HTTP_X_FILENAME'] != null) {
      if ($_SERVER['HTTP_X_FILENAME'] != $this->UPLOAD_NAME) {
        $this->UPLOAD_NAME = $_SERVER['HTTP_X_FILENAME'];
      }
    }
    $chunks = false;
    if (isset($_SERVER['HTTP_X_FILESIZE']) && $_SERVER['HTTP_X_FILESIZE'] != null) {
      if ($_SERVER['HTTP_X_FILESIZE'] > $this->UPLOAD_SIZE) {
        $chunks = true;
      }
    }
    $remove = false;
    if (isset($_SERVER['HTTP_X_REMOVE']) && $_SERVER['HTTP_X_REMOVE'] != null) {
      $remove = true;
    }
    if (dg::checkmem($this->UPLOAD_TMPNAME, 400, 400, false)) {
      $this->uploadDisplayError(
        JText::sprintf('COM_DATSOGALLERY_UPLOAD_MEMORY_LIMIT_ERROR',
          dg::checkmem($this->UPLOAD_TMPNAME, 400, 400, true))
      );
      $remove = true;
    }
    if (isset($this->UPLOAD_SETTINGS['maxFolderFiles'])
      && $this->UPLOAD_SETTINGS['maxFolderFiles'] > 0) {
      $user   = JFactory::getUser();
      $isroot = $user->authorise('core.admin');
      $tup    = dg::userimages();
      $aup    = !$isroot ? $cfg->{'acl_' . max($user->groups)} : 0;
      if (!$isroot && $tup == $aup && $aup != '-1' && !$this->UPLOAD_SETTINGS['avatarFilename']) {
        $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_MAXIMUM_NUMBER_OF_FILES_ALLOWED_REACHED'));
      }
    }
    $extension = strtolower(pathinfo($this->UPLOAD_NAME, PATHINFO_EXTENSION));
    if (array_search(strtolower($extension), array_map('strtolower', $this->UPLOAD_SETTINGS['extension'])) === false
      && count($this->UPLOAD_SETTINGS['extension']) > 0) {
      $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_FILE_EXTENSION_NOT_ALLOWED'));
    }
    if (strlen($this->UPLOAD_NAME) > 180) {
      $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_FILE_NAME_IS_TOO_LONG'));
    }
    if (($this->UPLOAD_SIZE > ($this->UPLOAD_SETTINGS['maxFileSize'] * 1024))
      || (isset($_SERVER['HTTP_X_FILESIZE'])
        && $_SERVER['HTTP_X_FILESIZE'] > ($this->UPLOAD_SETTINGS['maxFileSize'] * 1024))) {
      $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_FILE_SIZE_IS_TOO_BIG'));
    }
    if ($this->UPLOAD_SETTINGS['rename']) {
      $filename = $this->sanitizeFilename(basename($this->UPLOAD_NAME, '.' . $extension), $extension);
    } else {
      $filename = $this->UPLOAD_NAME;
    }
    if (!$this->UPLOAD_SETTINGS['overwrite']) {
      $i           = 1;
      $newfilename = $filename;
      while (file_exists($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $newfilename)) {
        $newfilename = basename($filename, '.' . $extension) . '(' . $i . ').' . $extension;
        $l           = strlen($this->UPLOAD_NAME);
        if ($l > 180) {
          substr($string, 0, (180 - $l));
          $newfilename = substr(basename($filename, '.' . $extension), 0, (180 - $l)) . '(' . $i . ').' . $extension;
        }
        $i++;
      }
      $filename = $newfilename;
    }

    if ($chunks) {
      if ($remove) {
        $s = unlink($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp');
      } else {
        file_put_contents($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp', file_get_contents($_FILES["file"]["tmp_name"]), FILE_APPEND);
        $resultMime = $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp';
        if (!in_array($this->get_mime_type($resultMime), $allowed_mime)) {
          unlink($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp');
          $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_INVALID_IMAGE_MIME_TYPE'));
        }
        $resultSize = filesize($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp');
        if ($resultSize > ($this->UPLOAD_SETTINGS['maxFileSize'] * 1024)) {
          unlink($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp');
          $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_FILE_SIZE_IS_TOO_BIG'));
        } else
        if ($_SERVER['HTTP_X_FILESIZE'] == $resultSize) {
          $this->checkImageSize($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp');
          if ($this->UPLOAD_SETTINGS['uniqueFilename']) {
            $n = $this->uniqueFilename($this->UPLOAD_SETTINGS['uploadFolder'], $extension);
            rename(($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp'), $n);
            $uploadedFile = $n;
          } else
          if ($this->UPLOAD_SETTINGS['avatarFilename']) {
            if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpg")) {
              unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpg");
            } else if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpeg")) {
              unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpeg");
            } else if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".png")) {
              unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".png");
            } else if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".gif")) {
              unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".gif");
            }
            $a = $this->avatarFilename($this->UPLOAD_SETTINGS['uploadFolder'], $extension);
            rename(($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp'), $a);
            $uploadedFile = $a;
            if ($this->UPLOAD_SETTINGS['resizeCrop']) {
              if ($this->resizeCrop(400, 400, $uploadedFile, $uploadedFile) === false) {
                $this->uploadDisplayError(
                  JText::sprintf('COM_DATSOGALLERY_UPLOAD_MEMORY_LIMIT_ERROR',
                    dg::checkmem($uploadedFile, 400, 400, true))
                );
              }
            }
          } else
          if ($this->UPLOAD_SETTINGS['changeFile']) {
            rename($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp', $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename);
            $uploadedFile = $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename;
          } else {
            rename($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename . '.tmp', $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename);
            $uploadedFile = $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename;
          }
        }
        $uploaded = true;
      }
    } else {
      move_uploaded_file($this->UPLOAD_TMPNAME, $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename);
      if (!in_array($this->get_mime_type($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename), $allowed_mime)) {
        unlink($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename);
        $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_INVALID_IMAGE_MIME_TYPE'));
      }
      $this->checkImageSize($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename);
      $uploadedFile = $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename;
      if ($this->UPLOAD_SETTINGS['uniqueFilename']) {
        $n = $this->uniqueFilename($this->UPLOAD_SETTINGS['uploadFolder'], $extension);
        rename(($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename), $n);
        $uploadedFile = $n;
      }
      if ($this->UPLOAD_SETTINGS['avatarFilename']) {
        if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpg")) {
          unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpg");
        } else if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpeg")) {
          unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".jpeg");
        } else if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".png")) {
          unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".png");
        } else if (file_exists($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".gif")) {
          unlink($this->avatarName($this->UPLOAD_SETTINGS['uploadFolder']) . ".gif");
        }
        $a = $this->avatarFilename($this->UPLOAD_SETTINGS['uploadFolder'], $extension);
        rename(($this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename), $a);
        $uploadedFile = $a;
        if ($this->UPLOAD_SETTINGS['resizeCrop']) {
          if ($this->resizeCrop(400, 400, $uploadedFile, $uploadedFile) === false) {
            $this->uploadDisplayError(
              JText::sprintf('COM_DATSOGALLERY_UPLOAD_MEMORY_LIMIT_ERROR',
                dg::checkmem($uploadedFile, 400, 400, true))
            );
          }
        }
      }
      if ($this->UPLOAD_SETTINGS['changeFile']) {
        $uploadedFile = $this->UPLOAD_SETTINGS['uploadFolder'] . '/' . $filename;
      }
      $uploaded = true;
    }

    if ($uploaded && $uploadedFile != '') {
      $json = array(
        'status' => JText::_('COM_DATSOGALLERY_FILE_UPLOADED')
      );
      if ($this->UPLOAD_SETTINGS['returnLocation']) {
        $json['url'] = $uploadedFile;
      }
      echo json_encode($json);
    } else {
      $json = array(
        'status' => JText::_('COM_DATSOGALLERY_UPLOADING')
      );
      echo json_encode($json);
    }
  }

  private function sanitizeFilename($filename, $extension)
  {
    $filename = preg_replace('/[^a-zA-Z0-9_-]/i', '_', $filename);
    if ($extension != '') {
      $filename .= '.' . $extension;
    }
    return $filename;
  }

  private function checkImageSize($file)
  {
    if ($this->UPLOAD_SETTINGS['maxWidth'] != 0 || $this->UPLOAD_SETTINGS['maxHeight'] != 0) {
      list($width, $height) = @getimagesize($file);
      if (($width > $this->UPLOAD_SETTINGS['maxWidth'] && $this->UPLOAD_SETTINGS['maxWidth'] > 0)
        && ($height > $this->UPLOAD_SETTINGS['maxHeight'] && $this->UPLOAD_SETTINGS['maxHeight'] > 0)) {
        $p = unlink($file);
        $this->uploadDisplayError(
          JText::sprintf('COM_DATSOGALLERY_IMAGE_CANNOT_EXCEEDS',
            $this->UPLOAD_SETTINGS['maxWidth'],
            $this->UPLOAD_SETTINGS['maxHeight'])
        );
      }
    }
  }

  private function get_mime_type($file)
  {
    $mtype = false;
    $info = @getimagesize($file);
    $mtype = $info['mime'];
    return $mtype;
  }

  private function uploadDisplayError($error)
  {
    $json = array('error' => $error);
    echo json_encode($json);
    exit();
  }

  private function uniqueFilename($folder, $extension)
  {
    $catid = $this->UPLOAD_SETTINGS['categoryId'];
    $uid1  = substr(strtoupper(md5(uniqid(time()))), 5, 12) . '-' . $catid;
    $file  = $folder . '/' . $uid1 . '.' . strtolower($extension);
    while (file_exists($file)) {
      $uid2 = substr(strtoupper(md5(uniqid(time()))), 5, 12) . '-' . $catid;
      $file = $folder . '/' . $uid2 . '.' . strtolower($extension);
    }
    return $file;
  }

  private function changeOriginal($folder, $extension)
  {
    $original = $this->UPLOAD_SETTINGS['originalName'];
    $file     = $folder . '/' . JFile::stripExt($original) . '.' . strtolower($extension);
    return $file;
  }

  private function avatarFilename($folder, $extension)
  {
    $user   = JFactory::getUser();
    $avatar = $user->id . '-avatar';
    $file   = $folder . '/' . $avatar . '.' . strtolower($extension);
    return $file;
  }

  private function avatarName($folder)
  {
    $user   = JFactory::getUser();
    $avatar = $user->id . '-avatar';
    $file   = $folder . '/' . $avatar;
    return $file;
  }

  private function resizeCrop($max_width, $max_height, $source_file, $dst_dir, $quality = 80)
  {
    if (dg::checkmem($source_file, $max_width, $max_height) === false) {
      return false;
    } else {
      $imgsize = @getimagesize($source_file);
      $width   = $imgsize[0];
      $height  = $imgsize[1];
      $mime    = $imgsize['mime'];
      switch ($mime) {
        case 'image/gif':
          $image_create = "imagecreatefromgif";
          $image        = "imagegif";
          break;

        case 'image/png':
          $image_create = "imagecreatefrompng";
          $image        = "imagepng";
          $quality      = 7;
          break;

        case 'image/jpeg':
          $image_create = "imagecreatefromjpeg";
          $image        = "imagejpeg";
          $quality      = 80;
          break;

        default:
          return false;
          break;
      }
      $dst_img    = imagecreatetruecolor($max_width, $max_height);
      $src_img    = $image_create($source_file);
      $width_new  = $height * $max_width / $max_height;
      $height_new = $width * $max_height / $max_width;

      imagealphablending($dst_img, false);
      imagesavealpha($dst_img, true);
      $transparent = imagecolorallocatealpha($dst_img, 255, 255, 255, 127);
      imagefilledrectangle($dst_img, 0, 0, $max_width, $max_height, $transparent);
      if ($width_new > $width) {
        $h_point = (($height - $height_new) / 2);
        imagecopyresampled($dst_img, $src_img, 0, 0, 0, $h_point, $max_width, $max_height, $width, $height_new);
      } else {
        $w_point = (($width - $width_new) / 2);
        imagecopyresampled($dst_img, $src_img, 0, 0, $w_point, 0, $max_width, $max_height, $width_new, $height);
      }
      $image($dst_img, $dst_dir, $quality);
      if ($dst_img) {
        imagedestroy($dst_img);
      }

      if ($src_img) {
        imagedestroy($src_img);
      }
      return true;
    }
  }

  private function rotateImage($file)
  {
    if (function_exists('exif_read_data')) {
      $exif        = @exif_read_data($file);
      $size        = @getimagesize($file);
      $exif_orient = isset($exif['Orientation']) ? $exif['Orientation'] : 0;
      $rotateImage = 0;
      if (6 == $exif_orient) {
        $rotateImage      = 90;
        $imageOrientation = 1;
      } elseif (3 == $exif_orient) {
        $rotateImage      = 180;
        $imageOrientation = 1;
      } elseif (8 == $exif_orient) {
        $rotateImage      = 270;
        $imageOrientation = 1;
      }
      if ($rotateImage) {
        $rotateImage = -$rotateImage;
        switch ($size['mime']) {
          case 'image/jpeg':
            $source = @imagecreatefromjpeg($file);
            $rotate = @imagerotate($source, $rotateImage, 0);
            @imagejpeg($rotate, $file, 100);
            break;

          case 'image/png':
            $source = @imagecreatefrompng($file);
            $rotate = @imagerotate($source, $rotateImage, 0);
            @imagepng($rotate, $file);
            break;

          case 'image/gif':
            $source = @imagecreatefromgif($file);
            $rotate = @imagerotate($source, $rotateImage, 0);
            @imagegif($rotate, $file);
            break;

          default:
            break;
        }
      }
    }
    return $file;
  }

  private function getUploadFolderSize()
  {
    $path  = $this->UPLOAD_SETTINGS['uploadFolder'];
    $limit = ($this->UPLOAD_SETTINGS['maxFolderSize'] * 1024);
    if ($limit > 0) {
      $size = 0;
      $path = realpath($path);
      if ($path !== false) {
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS)) as $object) {
          $size += $object->getSize();
        }
      }
      if ($size > $limit) {
        $this->uploadDisplayError(JText::_('COM_DATSOGALLERY_UPLOAD_FOLDER_MAXIMUM_SIZE_REACHED'));
      }
    }
  }
}
