<?php
  /**
   * @package    com_datsogallery
   * @author     Andrey Datso <support@datso.fr>
   * @copyright  (c) 2006 - 2020 Andrey Datso. All rights reserved.
   * @license    GNU General Public License version 2 or later; see LICENSE.txt
   */
  defined('_JEXEC') or die;
  define('MODIFIED', 1);
  define('NOT_MODIFIED', 2);

  class com_datsogalleryInstallerScript
  {

    public function preflight($type, $parent)
    {
      $app       = JFactory::getApplication();
      $jversion  = new JVersion;
      $manifest  = $parent->get("manifest");
      $release   = (string) $manifest['version'];
      $php       = '7.2';
      $is_apache = (strpos($_SERVER['SERVER_SOFTWARE'], 'Apache') !== false
        || strpos($_SERVER['SERVER_SOFTWARE'], 'LiteSpeed') !== false);
      $is_nginx = (strpos($_SERVER['SERVER_SOFTWARE'], 'nginx') !== false);
      if (!$jversion->isCompatible($release)) {
        $app->enqueueMessage(
          JText::sprintf('COM_DATSOGALLERY_COMPATIBILITY_JOOMLA_ERROR', $release), 'notice'
        );
        return false;
      }
      if (!$is_apache && !$is_nginx) {
        $app->enqueueMessage(
          JText::_('COM_DATSOGALLERY_COMPATIBILITY_SERVER_ERROR'), 'notice'
        );
        return false;
      }
      if (version_compare(phpversion(), $php, '<')) {
        $app->enqueueMessage(
          JText::sprintf('COM_DATSOGALLERY_COMPATIBILITY_PHP_ERROR', $php), 'notice'
        );
        return false;
      }
      $db_driver = strtolower(JFactory::getDbo()->name);
      if ($db_driver == 'pdomysql') {
        $app->enqueueMessage(
          JText::_('COM_DATSOGALLERY_COMPATIBILITY_PDOMYSQL_ERROR'), 'notice'
        );
        return false;
      }
      $db_compatibility = array('mysql', 'mysqli');
      if (!in_array($db_driver, $db_compatibility)) {
        $app->enqueueMessage(
          JText::_('COM_DATSOGALLERY_COMPATIBILITY_MYSQL_ERROR'), 'notice'
        );
        return false;
      }
      if (!extension_loaded('gd') && !extension_loaded('imagick')) {
        $app->enqueueMessage(
          JText::_('COM_DATSOGALLERY_COMPATIBILITY_GD_ERROR'), 'notice'
        );
        return false;
      }
      if (!function_exists('exif_read_data')) {
        $app->enqueueMessage(
          JText::_('COM_DATSOGALLERY_COMPATIBILITY_EXIF_ERROR'), 'notice'
        );
        return false;
      }
      return true;
    }

    public function postflight($type, $parent)
    {
      $app    = JFactory::getApplication();
      $rootId = $this->checkRootId();
      if (!$rootId) {
        $this->createRoot();
        $this->uninstall_old();
      }
      $this->cleanup();
      $this->folders();
      $this->compatibility_process();
      $this->addConfig($parent);
      $db    = JFactory::getDbo();
      $query = $db->getQuery(true)
        ->update('#__extensions')
        ->set('enabled = ' . $db->q(1))
        ->where('type = ' . $db->q('plugin'))
        ->where('element = ' . $db->q('datsogallery'))
        ->where('folder = ' . $db->q('installer'));
      $db->setQuery($query);
      try {
        $db->execute();
        $app->enqueueMessage(
          JText::sprintf('COM_DATSOGALLERY_POSTFLIGHT_MSG',
            '<strong><a href="index.php?option=com_datsogallery&view=settings">',
            '</a></strong>'), 'Notice'
        );
      } catch (Exception $ex) {
        $app->enqueueMessage(
          JText::_('There was an error enabling plugin DatsoGallery Installer!',
            $ex->getMessage()), 'error'
        );
      }
      $installation_folder = $parent->getParent()->getPath('source');
      $xml                 = simplexml_load_file($installation_folder . '/datsogallery.xml');
      $language            = JFactory::getLanguage();
      $code                = substr($language->get('tag'), 0, 2);
      $lang                = ($code == 'ru') ? 'ru/' : '';
    ?>
<style type="text/css">
    .dg-flex {
        display: flex;
        background-color: #F0F0F0;
        padding: 40px;
        overflow: hidden;
        color: #444;
        font-size: 16px;
        line-height: 26px;
        box-sizing: border-box;
        margin-bottom: 30px;
        align-items: center;
        border-radius: 3px;
        border: 1px solid #ddd;
    }
</style>
<div class="dg-flex">
    <div style="flex-basis:200px;padding-right:40px;">
        <img src="../administrator/components/com_datsogallery/assets/images/datsogallery_logo.svg" alt="Datso Gallery" />
    </div>
    <div style="flex-grow:1">
        <div>
            <h2>Datso Gallery <span><?=$xml->version;?></span></h2>
            <p>
                <?=JText::_('COM_DATSOGALLERY_SLOGAN');?>
            </p>
        </div>
        <div>
            <a href="https://www.datso.fr/<?=$lang;?>products/datsogallery/changelog.html" target="_blank" class="btn btn-small">
                <span class="icon-list-2"></span>
                <?=JText::_('COM_DATSOGALLERY_CHANGELOG');?>
            </a>
        </div>
    </div>
</div>
<?php
  }

    public function install($parent)
    {
      $source = $parent->getParent()->getPath('source');
      $this->installDb($parent);
      $this->addRoot();
      $installer = new JInstaller;
      $installer->install($source . '/plugins/installer/datsogallery/');
    }

    private function installDb($parent)
    {
      $installation_folder = $parent->getParent()->getPath('source');
      $app                 = JFactory::getApplication();
      if (function_exists('simplexml_load_file')
        && file_exists($installation_folder . '/installer/structure.xml')) {
        $component_data = simplexml_load_file($installation_folder . '/installer/structure.xml');
        foreach ($component_data->children() as $table) {
          $this->processTable($app, $table);
        }
      } else {
        if (!function_exists('simplexml_load_file')) {
          $app->enqueueMessage(
            JText::_('This script needs \'simplexml_load_file\' to update the component')
          );
        } else {
          $app->enqueueMessage(
            JText::_('Structure file was not found.')
          );
        }
      }
    }

    private function processTable($app, $table)
    {
      $db          = JFactory::getDbo();
      $table_added = false;
      if (isset($table['action'])) {
        switch ($table['action']) {
          case 'add':
            if (!$this->existsTable($table['table_name'])) {
              $create_statement = $this->generateCreateTableStatement($table);
              $db->setQuery($create_statement);
              try {
                $db->execute();
                $table_added = true;
                $this->setTableEngine((string) $table['table_name']);
                $db->alterTableCharacterSet((string) $table['table_name']);
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error creating the table `%s`. Error: %s',
                    (string) $table['table_name'], $ex->getMessage()), 'error'
                );
              }
            }
            break;

          case 'change':
            if ($this->existsTable($table['old_name'])
              && !$this->existsTable($table['new_name'])) {
              try {
                $db->renameTable($table['old_name'], $table['new_name']);
                $this->setTableEngine((string) $table['new_name']);
                $db->alterTableCharacterSet((string) $table['table_name']);
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error renaming the table `%s`. Error: %s',
                    $table['old_name'], $ex->getMessage()), 'error'
                );
              }
            } else {
              if (!$this->existsTable($table['table_name'])) {
                $create_statement = $this->generateCreateTableStatement($table);
                $db->setQuery($create_statement);
                try {
                  $db->execute();
                  $table_added = true;
                  $this->setTableEngine((string) $table['table_name']);
                  $db->alterTableCharacterSet((string) $table['table_name']);
                } catch (Exception $ex) {
                  $app->enqueueMessage(
                    JText::sprintf('There was an error creating the table `%s`. Error: %s',
                      $table['table_name'], $ex->getMessage()), 'error'
                  );
                }
              }
            }
            break;

          case 'remove':
            try {
              $db->dropTable((string) $table['table_name'], true);
            } catch (Exception $ex) {
              $app->enqueueMessage(
                JText::sprintf('There was an error deleting Table `%s`. Error: %s',
                  $table['table_name'], $ex->getMessage()), 'error'
              );
            }
            break;
        }
      }
      if (!$table_added) {
        if ($this->existsTable($table['table_name'])) {
          $this->setTableEngine((string) $table['table_name']);
          $db->alterTableCharacterSet((string) $table['table_name']);
          $this->executeFieldsUpdating($app, $table);
        }
      }
    }

    private function setTableEngine($table_name)
    {
      if ($this->getTableEngine($table_name) != 'InnoDB') {
        $app    = JFactory::getApplication();
        $db     = JFactory::getDbo();
        $engine = JText::sprintf('ALTER TABLE %s ENGINE = InnoDB', $table_name);
        $db->setQuery($engine);
        try {
          $db->execute();
        } catch (Exception $ex) {
          $app->enqueueMessage(
            JText::sprintf('There was an error modifying the table `%s` engine. Error: %s',
              $table_name, $ex->getMessage()), 'error'
          );
        }
      }
    }

    private function getTableEngine($table_name)
    {
      $db         = JFactory::getDbo();
      $table_name = str_replace('#__', $db->getPrefix(), (string) $table_name);
      $query      = $db->getQuery(true);
      $query      = 'SHOW TABLE STATUS WHERE Name = ' . $db->q($table_name);
      $db->setQuery($query);
      $obj = $db->loadObject();
      return $obj->Engine;
    }

    private function existsTable($table_name)
    {
      $db         = JFactory::getDbo();
      $table_name = str_replace('#__', $db->getPrefix(), (string) $table_name);
      return in_array($table_name, $db->getTableList());
    }

    private function generateCreateTableStatement($table)
    {
      $create_table_statement = '';
      if (isset($table->field)) {
        $fields             = $table->children();
        $fields_definitions = array();
        $indexes            = array();
        $db                 = JFactory::getDbo();
        foreach ($fields as $field) {
          $field_definition = $this->generateColumnDeclaration($field);
          if ($field_definition !== false) {
            $fields_definitions[] = $field_definition;
          }
          if ($field['index'] == 'index') {
            $indexes[] = $field['field_name'];
          }
        }
        foreach ($indexes as $index) {
          $fields_definitions[] = JText::sprintf('INDEX %s (%s ASC)',
            $db->qn((string) $index), $index);
        }
        if (!isset($table['no_key'])) {
          $fields_definitions[] = 'PRIMARY KEY (`id`)';
        }
        $create_table_statement = JText::sprintf('CREATE TABLE IF NOT EXISTS %s (%s)',
          $table['table_name'], implode(',', $fields_definitions));
      }
      return $create_table_statement;
    }

    private function generateColumnDeclaration($field)
    {
      $db        = JFactory::getDbo();
      $col_name  = $db->qn((string) $field['field_name']);
      $data_type = $this->getFieldType($field);
      if ($data_type !== false) {
        $default_value = (isset($field['default'])) ? 'DEFAULT ' . $field['default'] : '';
        $other_data    = '';
        if (isset($field['is_autoincrement']) && $field['is_autoincrement'] == '1') {
          $other_data .= ' AUTO_INCREMENT';
        }
        return JText::sprintf('%s %s NOT NULL %s %s', $col_name, $data_type,
          $default_value, $other_data);
      }
      return false;
    }

    private function getFieldAfter($field)
    {
      $data_type = '';
      if (isset($field['after'])) {
        $data_type .= ' AFTER ' . (string) $field['after'];
      }
      return (!empty($data_type)) ? $data_type : false;
    }

    private function getFieldType($field)
    {
      $data_type = (string) $field['field_type'];
      if (isset($field['field_length'])) {
        $data_type .= '(' . (string) $field['field_length'] . ')';
      }
      return (!empty($data_type)) ? $data_type : false;
    }

    private function executeFieldsUpdating($app, $table)
    {
      if (isset($table->field)) {
        foreach ($table->children() as $field) {
          $table_name = (string) $table['table_name'];
          $this->processField($app, $table_name, $field);
        }
      }
    }

    private function processField($app, $table_name, $field)
    {
      $db = JFactory::getDbo();
      if (isset($field['action'])) {
        switch ($field['action']) {
          case 'add':
            $result = $this->addField($table_name, $field);
            if ($result === MODIFIED) {
            } else {
              if ($result !== NOT_MODIFIED) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error adding the field `%s`. Error: %s',
                    $field['field_name'], $result), 'error'
                );
              }
            }
            if (isset($field['field_name']) && isset($field['after'])) {
              $position = JText::sprintf('ALTER TABLE %s CHANGE %s %s %s %s',
                $table_name, $field['field_name'], $field['field_name'],
                $this->getFieldType($field), $this->getFieldAfter($field));
              $db->setQuery($position);
              try {
                $db->execute();
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error moving the field `%s`. Error: %s',
                    $field['field_name'], $ex->getMessage()), 'error'
                );
              }
            }
            break;

          case 'change':
            if (isset($field['old_name']) && isset($field['new_name'])) {
              if ($this->existsField($table_name, $field['old_name'])) {
                $renaming_statement = JText::sprintf('ALTER TABLE %s CHANGE %s %s %s %s',
                  $table_name, $field['old_name'], $field['new_name'],
                  $this->getFieldType($field), $this->getFieldAfter($field));
                $db->setQuery($renaming_statement);
                try {
                  $db->execute();
                } catch (Exception $ex) {
                  $app->enqueueMessage(
                    JText::sprintf('There was an error renaming the field `%s`. Error: %s',
                      $field['field_name'], $ex->getMessage()), 'error'
                  );
                }
              } else {
                $result = $this->addField($table_name, $field);
                if ($result === MODIFIED) {
                } else {
                  if ($result !== NOT_MODIFIED) {
                    $app->enqueueMessage(
                      JText::sprintf('There was an error modifying the field `%s`. Error: %s',
                        $field['field_name'], $result), 'error'
                    );
                  }
                }
              }
            } else {
              $result = $this->addField($table_name, $field);
              if ($result === MODIFIED) {
              } else {
                if ($result !== NOT_MODIFIED) {
                  $app->enqueueMessage(
                    JText::sprintf('There was an error adding the field `%s`. Error: %s',
                      $field['field_name'], $result), 'error'
                  );
                }
              }
            }
            break;

          case 'remove':
            if ($this->existsField($table_name, $field['field_name'])) {
              $drop_statement = JText::sprintf('ALTER TABLE %s DROP COLUMN %s',
                $table_name, $field['field_name']);
              $db->setQuery($drop_statement);
              try {
                $db->execute();
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error deleting the field `%s`. Error: %s',
                    $field['field_name'], $ex->getMessage()), 'error'
                );
              }
            }
            break;

          case 'convert':
            if ($this->existsField($table_name, $field['old_name'])) {
              $converting_statement_add = JText::sprintf(
                'ALTER TABLE %s ADD COLUMN %s DATETIME NOT NULL DEFAULT '
                . $db->q('0000-00-00 00:00:00') . ' AFTER %s',
                $table_name, $field['new_name'], $field['after']
              );
              $db->setQuery($converting_statement_add);
              try {
                $db->execute();
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error adding the field `%s`. Error: %s',
                    $field['new_name'], $ex->getMessage()), 'error'
                );
              }
              $converting_statement_set = JText::sprintf(
                'UPDATE %s SET %s = FROM_UNIXTIME(%s)',
                $table_name, $field['new_name'], $field['old_name']
              );
              $db->setQuery($converting_statement_set);
              try {
                $db->execute();
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error converting the field `%s`. Error: %s',
                    $field['new_name'], $ex->getMessage()), 'error'
                );
              }
              $converting_statement_drop = JText::sprintf('ALTER TABLE %s DROP COLUMN %s',
                $table_name, $field['old_name']);
              $db->setQuery($converting_statement_drop);
              try {
                $db->execute();
              } catch (Exception $ex) {
                $app->enqueueMessage(
                  JText::sprintf('There was an error deleting the field `%s`. Error: %s',
                    $field['old_name'], $ex->getMessage()), 'error'
                );
              }
            }
            break;
        }
      } else {
        $result = $this->addField($table_name, $field);
        if ($result === MODIFIED) {
        } else {
          if ($result !== NOT_MODIFIED) {
            $app->enqueueMessage(
              JText::sprintf('There was an error adding the field `%s`. Error: %s',
                $field['field_name'], $result), 'error'
            );
          }
        }
      }
    }

    private function addField($table_name, $field)
    {
      $app             = JFactory::getApplication();
      $db              = JFactory::getDbo();
      $query_generated = false;
      if ($this->existsField($table_name, $field['field_name'])) {
        if (isset($field['index'])
          && $field['index'] == 'index'
          && !$this->existsIndex($table_name, $field['field_name'])) {
          $db->setQuery("ALTER TABLE `" . $table_name . "` ADD INDEX(`" . $field['field_name'] . "`)");
          $db->execute();
        }
        if ($this->needsToUpdate($table_name, $field)) {
          $change_statement = $this->generateChangeFieldStatement($table_name, $field);
          $db->setQuery($change_statement);
          $query_generated = true;
        }
      } else {
        $add_statement = $this->generateAddFieldStatement($table_name, $field);
        $db->setQuery($add_statement);
        $query_generated = true;
      }
      if ($query_generated) {
        try {
          $db->execute();
          return MODIFIED;
        } catch (Exception $ex) {
          return $ex->getMessage();
        }
      }
      return NOT_MODIFIED;
    }

    private function existsField($table_name, $field_name)
    {
      $db = JFactory::getDbo();
      return in_array($field_name, array_keys($db->getTableColumns($table_name)));
    }

    private function needsToUpdate($table_name, $field)
    {
      $db    = JFactory::getDbo();
      $table = str_replace('#__', $db->getPrefix(), (string) $table_name);
      $query = $db->getQuery(true);
      $query->select('COLUMN_TYPE')
        ->from($db->qn('INFORMATION_SCHEMA.COLUMNS'))
        ->where(
          array(
            $db->qn('TABLE_NAME') . ' = ' . $db->q($table),
            $db->qn('COLUMN_NAME') . ' = ' . $db->q($field['field_name'])
          )
        );
      $db->setQuery($query);
      $field_type = $db->loadResult();
      $type       = preg_replace('/[(0-9\.,)]/', '', $field_type);
      if ($type == strtolower(strtok($field['field_type'], "(, "))) {
        return false;
      }
      if (isset($field['field_length'])) {
        preg_match('/\((.*?)\)/', $field_type, $match);
        if (!empty($match[1]) && (int) $match[1] == (int) $field['field_length']) {
          return false;
        }
      }
      return true;
    }

    private function existsIndex($table_name, $field_name)
    {
      $db = JFactory::getDbo();
  //$table = str_replace('#__', $db->getPrefix(), (string) $table_name);
      $query = $db->getQuery(true);
      $query = 'SHOW INDEX FROM ' . $table_name . ' WHERE Column_name = ' . $db->q($field_name);
      $db->setQuery($query);
      $db->execute();
      $obj = $db->loadObject() ? $db->loadObject()->Non_unique : false;
      return $obj;
    }

    private function generateChangeFieldStatement($table_name, $field)
    {
      $column_declaration = $this->generateColumnDeclaration($field);
      return JText::sprintf('ALTER TABLE %s MODIFY %s', $table_name, $column_declaration);
    }

    private function generateAddFieldStatement($table_name, $field)
    {
      $column_declaration = $this->generateColumnDeclaration($field);
      return JText::sprintf('ALTER TABLE %s ADD %s', $table_name, $column_declaration);
    }

    public function update($parent)
    {
      $source = $parent->getParent()->getPath('source');
      $this->installDb($parent);
      $installer = new JInstaller;
      $installer->install($source . '/plugins/installer/datsogallery/');
    }

    public function uninstall($parent)
    {
      $this->delTables();
      $this->delMenu();
    }

    public function delTables()
    {
      $app    = JFactory::getApplication();
      $db     = JFactory::getDbo();
      $tables = array('#__datsogallery_basket',
        '#__datsogallery_blacklist',
        '#__datsogallery_categories',
        '#__datsogallery_colors',
        '#__datsogallery_comments',
        '#__datsogallery_comment_likes',
        '#__datsogallery_downloads',
        '#__datsogallery_favorites',
        '#__datsogallery_hits',
        '#__datsogallery_images',
        '#__datsogallery_likes',
        '#__datsogallery_members',
        '#__datsogallery_purchases',
        '#__datsogallery_reports',
        '#__datsogallery_settings',
        '#__datsogallery_tags');
      foreach ($tables as $table) {
        try {
          $db->dropTable((string) $table, true);
        } catch (Exception $ex) {
          $app->enqueueMessage(
            JText::sprintf('There was an error deleting Table `%s`. Error: %s',
              $table, $ex->getMessage()), 'error'
          );
        }
      }
    }

    public function delMenu()
    {
      $app   = JFactory::getApplication();
      $db    = JFactory::getDbo();
      $query = $db->getQuery(true);
      $query->delete($db->qn('#__menu'));
      $query->where($db->qn('link') . ' LIKE ' . $db->q('index.php?option=com_datsogallery%'));
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $ex) {
        $app->enqueueMessage(
          JText::sprintf('An error occurred while deleting menu items related to DatsoGallery',
            $ex->getMessage()), 'error'
        );
      }
    }

    private function checkRootId()
    {
      $db    = JFactory::getDbo();
      $query = $db->getQuery(true);
      $query->select('id')
        ->from($db->qn('#__datsogallery_categories'))
        ->where($db->qn('id') . ' = 1')
        ->where($db->qn('parent_id') . ' = 0')
        ->where($db->qn('alias') . ' = ' . $db->q('root'));
      $db->setQuery($query);
      if ($db->loadResult()) {
        return true;
      } else {
        return false;
      }
    }

    public function addConfig($parent)
    {
      $app    = JFactory::getApplication();
      $db     = JFactory::getDbo();
      $source = $parent->getParent()->getPath('source');
      $query  = $db->getQuery(true);
      $query->select('name')->from($db->qn('#__datsogallery_settings'));
      $db->setQuery($query);
      $names = $db->loadColumn();
      $xml   = simplexml_load_file($source . '/admin/models/forms/settings.xml');
      foreach ($xml->children() as $fields) {
        foreach ($fields as $field) {
          if (in_array($field['name'], $names)) {
            continue;
          }

          if (in_array($field['name'], array('restoring', 'rules'))) {
            continue;
          }

          $name              = (string) $field['name'];
          $value             = (string) $field['default'];
          $new_record        = new stdClass();
          $new_record->name  = $name;
          $new_record->value = $value;
          $db->insertObject('#__datsogallery_settings', $new_record);
        }
      }
      return true;
    }

    public function addRoot()
    {
      $app     = JFactory::getApplication();
      $db      = JFactory::getDbo();
      $columns = array('id', 'asset_id', 'parent_id', 'lft', 'rgt', 'level', 'path',
        'ordering', 'title', 'alias', 'note', 'description', 'published',
        'approved', 'publish_up', 'publish_down', 'checked_out',
        'checked_out_time', 'access', 'params', 'metadesc', 'metakey',
        'metadata', 'created_by', 'created_time', 'modified_by',
        'modified_time', 'hits', 'language', 'cover', 'image_id'
      );
      $values = array(1, 0, 0, 0, 1, 0, $db->q(''), 1, $db->q('ROOT'), $db->q('root'),
        $db->q(''), $db->q(''), 1, 0, $db->q('0000-00-00 00:00:00'),
        $db->q('0000-00-00 00:00:00'), 0, $db->q('0000-00-00 00:00:00'), 1,
        $db->q('{}'), $db->q(''), $db->q(''), $db->q('{}'), 62,
        $db->q('2011-01-01 00:00:01'), 0, $db->q('0000-00-00 00:00:00'), 0,
        $db->q('*'), 0, 0
      );
      $query = $db->getQuery(true);
      $query->insert('#__datsogallery_categories')
        ->columns($db->qn($columns))
        ->values(implode(',', $values));
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $ex) {
        $app->enqueueMessage(
          JText::sprintf('An error occurred while creating ROOT category',
            $ex->getMessage()), 'error'
        );
      }
    }

    public function createRoot()
    {
      $app    = JFactory::getApplication();
      $db     = JFactory::getDbo();
      $squery = $db->getQuery(true);
      $query  = $db->getQuery(true);
      $squery->select('1')
        ->from($db->qn('#__datsogallery_categories', 'b'))
        ->where($db->qn('b.id') . ' = a.id + 1');
      $query->select('min(a.id + 1) as gap')
        ->from($db->qn('#__datsogallery_categories', 'a'))
        ->where('not exists (' . $squery . ')');
      $db->setQuery($query);
      $gapid = $db->loadResult();
      $query = $db->getQuery(true);
      $query->update($db->qn('#__datsogallery_categories'))
        ->set($db->qn('id') . ' = ' . $gapid)
        ->where($db->qn('id') . ' = 1')
        ->order($db->qn('id') . ' desc');
      $db->setQuery(str_replace('UPDATE', 'UPDATE IGNORE', $query));
      try {
        $db->execute();
      } catch (Exception $ex) {
        $app->enqueueMessage(
          JText::sprintf('An error occurred while reassigning one of the categories',
            $ex->getMessage()), 'error'
        );
      }
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_images')
        ->set('catid = ' . $gapid)
        ->where('catid = 1');
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $ex) {
        $app->enqueueMessage(
          JText::sprintf('An error occurred while reassigning a category for images',
            $ex->getMessage()), 'error'
        );
      }
      $this->addRoot();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_categories')
        ->set('parent_id = 1')
        ->where('parent_id = 0')
        ->where('id != 1');
      $db->setQuery($query);
      if (!$db->execute()) {
        return false;
      }
      JTable::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_datsogallery/tables');
      $table = JTable::getInstance('Category', 'DatsogalleryTable');
      if (!$table->rebuild()) {
        $app->enqueueMessage(
          JText::sprintf('An error occurred while rebuilding categories',
            $table->getError()), 'error'
        );
        return false;
      }
      $query = $db->getQuery(true);
      $query->select('*')
        ->from($db->qn('#__datsogallery_votes'))
        ->where($db->qn('vsum') . ' in (1,5)')
        ->order('vdate');
      $db->setQuery($query);
      $likes   = $db->loadObjectList();
      $columns = array('id', 'image_id', 'user_id', 'ip', 'rate', 'rated_on');
      foreach ($likes as $i => $like) {
        $i        = $i + 1;
        $vsum     = $like->vsum == 5 ? '1' : '2';
        $values[] = $i . ',' . $like->vpic . ',' . $like->user_id . ',' . $db->q(strtok($like->vip, ','))
        . ',' . $vsum . ',' . $db->q($like->vdate);
      }
      $query = $db->getQuery(true);
      $query->insert($db->qn('#__datsogallery_likes'))
        ->columns($db->qn($columns))
        ->values(implode('),(', $values));
      $db->setQuery($query);
      try {
        $db->execute();
      } catch (Exception $ex) {
        $app->enqueueMessage(
          JText::sprintf('An error occurred while votes conversion',
            $ex->getMessage()), 'error'
        );
      }
      $query = $db->getQuery(true);
      $query->select('*')
        ->from($db->qn('#__datsogallery_comments'))
        ->where($db->qn('text') . ' rlike ' . $db->q('<|/>| />'));
      $db->setQuery($query);
      $comments = $db->loadObjectList();
      foreach ($comments as $comment) {
        $query->clear()
          ->update('#__datsogallery_comments')
          ->set($db->qn('text') . ' = ' . $db->q(strip_tags($comment->text)))
          ->where($db->qn('id') . ' = ' . $comment->id);
        $db->setQuery($query);
        try {
          $db->execute();
        } catch (Exception $ex) {
          $app->enqueueMessage(
            JText::sprintf('An error occurred while cleanning comments',
              $ex->getMessage()), 'error'
          );
        }
      }
      $db->dropTable('#__datsogallery_votes');
      return true;
    }

    protected $outdated_files = array(
  // admin
      '/administrator/components/com_datsogallery/admin.datsogallery.html.php',
      '/administrator/components/com_datsogallery/admin.datsogallery.php',
      '/administrator/components/com_datsogallery/class.datsogallery.php',
      '/administrator/components/com_datsogallery/config.datsogallery.php',
      '/administrator/components/com_datsogallery/images.datsogallery.php',
      '/administrator/components/com_datsogallery/install.datsogallery.php',
      '/administrator/components/com_datsogallery/install.datsogallery.sql',
      '/administrator/components/com_datsogallery/uninstall.datsogallery.php',
      '/administrator/components/com_datsogallery/uninstall.datsogallery.sql',
      '/administrator/components/com_datsogallery/assets/css/bootstrap-wysihtml5.css',
      '/administrator/components/com_datsogallery/assets/css/bootstrap.css',
      '/administrator/components/com_datsogallery/assets/css/metisMenu.css',
      '/administrator/components/com_datsogallery/assets/css/sb-admin-2.css',
      '/administrator/components/com_datsogallery/assets/css/select2.css',
      '/administrator/components/com_datsogallery/assets/css/sky-tabs.css',
      '/administrator/components/com_datsogallery/assets/css/timeline.css',
      '/administrator/components/com_datsogallery/assets/css/uikit-rtl.css',
      '/administrator/components/com_datsogallery/assets/css/uikit.css',
      '/administrator/components/com_datsogallery/assets/js/bootstrap-wysihtml5.js',
      '/administrator/components/com_datsogallery/assets/js/chart.min.js',
      '/administrator/components/com_datsogallery/assets/js/classie.js',
      '/administrator/components/com_datsogallery/assets/js/jquery.lazyload.js',
      '/administrator/components/com_datsogallery/assets/js/modernizr.custom.js',
      '/administrator/components/com_datsogallery/assets/js/chart.js',
      '/administrator/components/com_datsogallery/assets/images/component-box.png',
      '/administrator/components/com_datsogallery/controllers/sanitize.php',
      '/administrator/components/com_datsogallery/controllers/basket.php',
      '/administrator/components/com_datsogallery/controllers/baskets.php',
      '/administrator/components/com_datsogallery/controllers/download.php',
      '/administrator/components/com_datsogallery/controllers/downloads.php',
      '/administrator/components/com_datsogallery/controllers/hit.php',
      '/administrator/components/com_datsogallery/controllers/hits.php',
      '/administrator/components/com_datsogallery/controllers/purchase.php',
      '/administrator/components/com_datsogallery/controllers/purchases.php',
      '/administrator/components/com_datsogallery/controllers/tag.php',
      '/administrator/components/com_datsogallery/controllers/tags.php',
      '/administrator/components/com_datsogallery/controllers/vote.php',
      '/administrator/components/com_datsogallery/controllers/votes.php',
      '/administrator/components/com_datsogallery/models/fields/categoryedit.php',
      '/administrator/components/com_datsogallery/models/fields/cmultiple.php',
      '/administrator/components/com_datsogallery/models/fields/createdby.php',
      '/administrator/components/com_datsogallery/models/fields/custom_field.php',
      '/administrator/components/com_datsogallery/models/fields/multi.php',
      '/administrator/components/com_datsogallery/models/fields/multi_categories.php',
      '/administrator/components/com_datsogallery/models/fields/timecreated.php',
      '/administrator/components/com_datsogallery/models/fields/timeupdated.php',
      '/administrator/components/com_datsogallery/models/forms/basket.xml',
      '/administrator/components/com_datsogallery/models/forms/download.xml',
      '/administrator/components/com_datsogallery/models/forms/hit.xml',
      '/administrator/components/com_datsogallery/models/forms/purchase.xml',
      '/administrator/components/com_datsogallery/models/forms/tag.xml',
      '/administrator/components/com_datsogallery/models/forms/vote.xml',
      '/administrator/components/com_datsogallery/models/basket.php',
      '/administrator/components/com_datsogallery/models/baskets.php',
      '/administrator/components/com_datsogallery/models/download.php',
      '/administrator/components/com_datsogallery/models/downloads.php',
      '/administrator/components/com_datsogallery/models/favorite.php',
      '/administrator/components/com_datsogallery/models/favorites.php',
      '/administrator/components/com_datsogallery/models/hit.php',
      '/administrator/components/com_datsogallery/models/hits.php',
      '/administrator/components/com_datsogallery/models/purchase.php',
      '/administrator/components/com_datsogallery/models/purchases.php',
      '/administrator/components/com_datsogallery/models/tag.php',
      '/administrator/components/com_datsogallery/models/tags.php',
      '/administrator/components/com_datsogallery/models/vote.php',
      '/administrator/components/com_datsogallery/models/votes.php',
      '/administrator/components/com_datsogallery/tables/basket.php',
      '/administrator/components/com_datsogallery/tables/download.php',
      '/administrator/components/com_datsogallery/tables/favorite.php',
      '/administrator/components/com_datsogallery/tables/hit.php',
      '/administrator/components/com_datsogallery/tables/purchase.php',
      '/administrator/components/com_datsogallery/tables/vote.php',
      '/administrator/components/com_datsogallery/helpers/image.php',
      '/administrator/components/com_datsogallery/helpers/exception.php',
      '/administrator/components/com_datsogallery/helpers/pclzip.lib.php',
      '/administrator/components/com_datsogallery/helpers/phpmailer.php',
      '/administrator/components/com_datsogallery/helpers/smtp.php',
  // site
      '/components/com_datsogallery/datso.ajax.php',
      '/components/com_datsogallery/datso.functions.php',
      '/components/com_datsogallery/controllers/datso.tmpimg.php',
      '/components/com_datsogallery/controllers/sanitize.php',
      '/components/com_datsogallery/controllers/categoryform.php',
      '/components/com_datsogallery/controllers/imageform.php',
      '/components/com_datsogallery/controllers/purchase.php',
      '/components/com_datsogallery/controllers/purchaseform.php',
      '/components/com_datsogallery/controllers/purchases.php',
      '/components/com_datsogallery/controllers/startpage.php',
      '/components/com_datsogallery/helpers/html/pagination.php',
      '/components/com_datsogallery/helpers/avatars.php',
      '/components/com_datsogallery/helpers/pagination.php',
      '/components/com_datsogallery/helpers/route.php',
      '/components/com_datsogallery/models/fields/createdby.php',
      '/components/com_datsogallery/models/fields/filemultiple.php',
      '/components/com_datsogallery/models/fields/foreignkey.php',
      '/components/com_datsogallery/models/fields/modifiedby.php',
      '/components/com_datsogallery/models/fields/submit.php',
      '/components/com_datsogallery/models/fields/timecreated.php',
      '/components/com_datsogallery/models/fields/timeupdated.php',
      '/components/com_datsogallery/models/forms/category.xml',
      '/components/com_datsogallery/models/forms/categoryform.xml',
      '/components/com_datsogallery/models/forms/filter_category.xml',
      '/components/com_datsogallery/models/forms/image.xml',
      '/components/com_datsogallery/models/forms/imageform.xml',
      '/components/com_datsogallery/models/forms/member.xml',
      '/components/com_datsogallery/models/forms/purchase.xml',
      '/components/com_datsogallery/models/forms/purchaseform.xml',
      '/components/com_datsogallery/models/forms/upload.xml',
      '/components/com_datsogallery/models/categoryform.php',
      '/components/com_datsogallery/models/imageform.php',
      '/components/com_datsogallery/models/purchase.php',
      '/components/com_datsogallery/models/purchaseform.php',
      '/components/com_datsogallery/models/purchases.php',
      '/components/com_datsogallery/models/startpage.php',
      '/components/com_datsogallery/views/member/tmpl/default.xml',
      '/components/com_datsogallery/views/member/tmpl/default_comments.php',
      '/components/com_datsogallery/assets/css/category-view.css',
      '/components/com_datsogallery/assets/css/comments-view.css',
      '/components/com_datsogallery/assets/css/hint.css',
      '/components/com_datsogallery/assets/css/member-view.css',
      '/components/com_datsogallery/assets/css/pagination.css',
      '/components/com_datsogallery/assets/css/shadowbox.css',
      '/components/com_datsogallery/assets/css/stacks.css',
      '/components/com_datsogallery/assets/css/steps.css',
      '/components/com_datsogallery/assets/css/style.css',
      '/components/com_datsogallery/assets/css/upload.css',
      '/components/com_datsogallery/assets/css/dg-ui.css',
      '/components/com_datsogallery/assets/css/dg-ui-black.css',
      '/components/com_datsogallery/assets/css/emoji.css',
      '/components/com_datsogallery/assets/css/fotorama.dev.css',
      '/components/com_datsogallery/assets/css/fotorama.png',
      '/components/com_datsogallery/assets/css/fotorama@2x.png',
      '/components/com_datsogallery/assets/fonts/dg-emoji.eot',
      '/components/com_datsogallery/assets/fonts/dg-emoji.svg',
      '/components/com_datsogallery/assets/fonts/dg-emoji.ttf',
      '/components/com_datsogallery/assets/fonts/dg-emoji.woff',
      '/components/com_datsogallery/assets/icons/dg-icons.eot',
      '/components/com_datsogallery/assets/js/datsogallery.min.js',
      '/components/com_datsogallery/assets/js/fotorama.dev.js',
      '/components/com_datsogallery/assets/js/isotope.pkgd.min.js',
      '/components/com_datsogallery/assets/js/jquery.autosize.min.js',
      '/components/com_datsogallery/assets/js/jquery.form.min.js',
      '/components/com_datsogallery/assets/js/jquery.imageloader.js',
      '/components/com_datsogallery/assets/js/jquery.steps.js',
      '/components/com_datsogallery/assets/js/jquery.validate.min.js',
      '/components/com_datsogallery/assets/js/jquery-ias.min.js',
      '/components/com_datsogallery/assets/js/jquery.touchSwipe.min.js',
      '/components/com_datsogallery/assets/js/shadowbox.js',
      '/components/com_datsogallery/assets/js/dgload.js',
      '/components/com_datsogallery/assets/js/gmaps.js',
      '/components/com_datsogallery/assets/js/grid.js',
      '/components/com_datsogallery/assets/js/imagesloaded.min.js',
      '/components/com_datsogallery/assets/js/jquery.alerts.js',
      '/components/com_datsogallery/assets/js/jquery.geocomplete.min.js',
      '/components/com_datsogallery/assets/js/jquery.tagsinput.min.js',
      '/components/com_datsogallery/assets/js/lib.js',
      '/components/com_datsogallery/assets/icons/shadowbox.eot',
      '/components/com_datsogallery/assets/icons/shadowbox.svg',
      '/components/com_datsogallery/assets/icons/shadowbox.ttf',
      '/components/com_datsogallery/assets/icons/shadowbox.woff',
      '/components/com_datsogallery/assets/images/sb_close.svg',
      '/components/com_datsogallery/assets/images/sb_next.svg',
      '/components/com_datsogallery/assets/images/sb_pause.svg',
      '/components/com_datsogallery/assets/images/sb_play.svg',
      '/components/com_datsogallery/assets/images/sb_previous.svg'
    );

    protected $outdated_folders = array(
  // admin
      '/administrator/components/com_datsogallery/css',
      '/administrator/components/com_datsogallery/fields',
      '/administrator/components/com_datsogallery/images',
      '/administrator/components/com_datsogallery/includes',
      '/administrator/components/com_datsogallery/js',
      '/administrator/components/com_datsogallery/js/locales',
      '/administrator/components/com_datsogallery/views/basket',
      '/administrator/components/com_datsogallery/views/baskets',
      '/administrator/components/com_datsogallery/views/download',
      '/administrator/components/com_datsogallery/views/downloads',
      '/administrator/components/com_datsogallery/views/favorite',
      '/administrator/components/com_datsogallery/views/favorites',
      '/administrator/components/com_datsogallery/views/hit',
      '/administrator/components/com_datsogallery/views/hits',
      '/administrator/components/com_datsogallery/views/purchase',
      '/administrator/components/com_datsogallery/views/purchases',
      '/administrator/components/com_datsogallery/views/tag',
      '/administrator/components/com_datsogallery/views/tags',
      '/administrator/components/com_datsogallery/views/vote',
      '/administrator/components/com_datsogallery/views/votes',
  // site
      '/components/com_datsogallery/assets/js/language',
      '/components/com_datsogallery/assets/gridzy',
      '/components/com_datsogallery/captcha',
      '/components/com_datsogallery/css',
      '/components/com_datsogallery/fonts',
      '/components/com_datsogallery/images',
      '/components/com_datsogallery/includes',
      '/components/com_datsogallery/libraries',
      '/components/com_datsogallery/views/datsogallery',
      '/components/com_datsogallery/views/showupload',
      '/components/com_datsogallery/views/startpage',
      '/zipimport'
    );

    protected function cleanup()
    {
      $app = JFactory::getApplication();
      if (!JFile::exists(JPATH_SITE . '/components/com_datsogallery/assets/images/watermark.png')) {
        JFile::move(JPATH_SITE . '/components/com_datsogallery/assets/images/watermark_bak.png',
          JPATH_SITE . '/components/com_datsogallery/assets/images/watermark.png');
      }
      $failed = array('outdated/unused file/folder deletion failed:');
      foreach ($this->outdated_files as $file) {
        if (JFile::exists(JPATH_SITE . $file) && !JFile::delete(JPATH_SITE . $file)) {
          $failed[] = $file;
        }
      }
      foreach ($this->outdated_folders as $folder) {
        if (JFolder::exists(JPATH_SITE . $folder) && !JFolder::delete(JPATH_SITE . $folder)) {
          $failed[] = $folder;
        }
      }
      if (count($failed) > 1) {
        $failed[] = 'please delete this file/folder manually';
        $app->enqueueMessage(implode('<br />', $failed), 'warning');
      }
    }

    protected function compatibility_check($prefix = null, $group = null, $name = null)
    {
      $path  = $prefix == 'plg' ? JPATH_PLUGINS . '/' . $group . '/' : JPATH_SITE . '/modules/';
      $type  = $prefix == 'plg' ? 'plugin' : 'module';
      $name  = $prefix == 'mod' ? implode('_', array($prefix, $group, $name)) : $name;
      $group = $prefix == 'mod' ? '' : $group;
      $app   = JFactory::getApplication();
      $db    = JFactory::getDbo();
      if (function_exists('simplexml_load_file')
        && file_exists($path . $name . '/' . $name . '.xml')) {
        $xml = simplexml_load_file($path . $name . '/' . $name . '.xml');
        if (($xml->version < '3.5.0') || ($name == 'dgcleaner' && $xml->version < '3.5.7')) {
          $app->enqueueMessage(
            JText::_('COM_DATSOGALLERY_DISABLED_EXTENSIONS_MSG'), 'warning'
          );
          $query = $db->getQuery(true)
            ->update('#__extensions')
            ->set('enabled = ' . $db->q(0))
            ->where('type = ' . $db->q($type))
            ->where('element = ' . $db->q($name))
            ->where('folder = ' . $db->q($group));
          $db->setQuery($query);
          try {
            $db->execute();
            $app->enqueueMessage(
              JText::sprintf('COM_DATSOGALLERY_OUTDATED_EXTENSION', strtolower($xml->name), strtolower($xml->version)), 'warning'
            );
          } catch (Exception $ex) {
            $app->enqueueMessage(
              JText::sprintf('COM_DATSOGALLERY_ERROR_DISABLING_EXTENSION',
                strtolower($xml->name),
                $ex->getMessage()
              ), 'error'
            );
          }
        }
      } else {
        if (!function_exists('simplexml_load_file')) {
          $app->enqueueMessage(
            JText::sprintf('COM_DATSOGALLERY_NEEDS_SIMPLEXML_LOAD_FILE', $name), 'error'
          );
        }
      }
    }

    protected $dg_extensions = array(
      'mod_datsogallery_cloudimages',
      'mod_datsogallery_comments',
      'mod_datsogallery_slideshow',
      'mod_datsogallery_stats',
      'mod_datsogallery_tagcloud',
      'mod_datsogallery_tree',
      'mod_datsogallery_ultimate',
      'plg_datsogallery_jomsocial',
      'plg_osmap_com_datsogallery',
      'plg_search_datsogallery',
      'plg_system_datsoredirect',
      'plg_system_dgcleaner',
      'plg_xmap_com_datsogallery'
    );

    protected function compatibility_process()
    {
      foreach ($this->dg_extensions as $extension) {
        $part = explode('_', $extension);
        echo $this->compatibility_check($part[0], $part[1], $part[2]);
      }
    }

    protected $create_folders = array(
      '/images/dg_originals',
      '/images/dg_avatars',
      '/images/dg_images',
      '/images/dg_images_webp',
    );

    protected function folders()
    {
      $app     = JFactory::getApplication();
      $failed  = array('Folder creation failed: ');
      $content = '<!DOCTYPE html><title></title>';
      foreach ($this->create_folders as $folder) {
        if (!JFolder::exists(JPATH_SITE . $folder)
          && !JFolder::create(JPATH_SITE . $folder)
          && !JFile::write(JPATH_SITE . $folder . '/index.html', $content)) {
          $failed[] = $folder;
        }
      }
      if (count($failed) > 1) {
        $failed[] = 'please create this folder manually';
        $app->enqueueMessage(implode('<br />', $failed), 'warning');
      }
    }

    public function uninstall_old()
    {
      $db    = JFactory::getDbo();
      $query = $db->getQuery(true)
        ->select('e.extension_id')
        ->from('#__extensions AS e')
        ->where('e.type = ' . $db->q('plugin'))
        ->where('(e.element like ' . $db->q('%datsogallery%')
          . ' AND e.folder = ' . $db->q('search') . ') OR (e.element like ' . $db->q('%datsogallery%')
          . ' AND e.folder = ' . $db->q('system') . ')');
      $db->setQuery($query);
      try {
        $plugins = $db->loadColumn();
      } catch (RuntimeException $e) {
        return false;
      }
      if (!empty($plugins)) {
        foreach ($plugins as $plugin) {
          $installer = new JInstaller;
          $installer->uninstall('plugin', $plugin);
        }
      }
      $query = $db->getQuery(true)
        ->select('e.extension_id')
        ->from('#__extensions AS e')
        ->where('e.type = ' . $db->q('module'))
        ->where('e.element like ' . $db->q('%datsogallery%'));
      $db->setQuery($query);
      try {
        $modules = $db->loadColumn();
      } catch (RuntimeException $e) {
        return false;
      }
      if (!empty($modules)) {
        foreach ($modules as $module) {
          $installer = new JInstaller;
          $installer->uninstall('module', $module);
        }
      }
      return true;
    }
  }
