<?php
/**
 * @package    com_datsogallery
 * @author     Andrey Datso <support@datso.fr>
 * @copyright  (c) 2006 - 2020 Andrey Datso. All rights reserved.
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

use Joomla\Registry\Registry;
use Joomla\String\StringHelper;
use Joomla\Utilities\ArrayHelper;

class DatsogalleryModelCategory extends JModelAdmin {

  protected $text_prefix = 'COM_DATSOGALLERY';

  public function __construct($config = array()) {
    parent::__construct($config);
    $this->batch_commands['flip_ordering'] = 'batchFlipordering';
  }

  public function getTable($type = 'Category', $prefix = 'DatsogalleryTable', $config = array()) {
    return JTable::getInstance($type, $prefix, $config);
  }

  public function getItem($pk = null) {
    if ($result = parent::getItem($pk)) {
      if ($result->id == 1) {
        $result->title = JText::_('COM_DATSOGALLERY_ROOT');
        $result->alias = JApplication::stringURLSafe(JText::_('COM_DATSOGALLERY_ROOT'));
      }
      if (!empty($result->access)) {
        $result->access = explode(',', $result->access);
      }
    }
    return $result;
  }

  public function getForm($data = array(), $loadData = true) {
    $id   = JFactory::getApplication()->input->get('id', 0);
    $form = $this->loadForm('com_datsogallery.category', 'category', array('control' => 'jform', 'load_data' => $loadData));
    if (empty($form)) {
      return false;
    }

    if ($id == 1) {
      $form->setFieldAttribute('title', 'readonly', 'true');
      $form->setFieldAttribute('alias', 'disabled', 'true');
      $form->setFieldAttribute('parent_id', 'disabled', 'true');
      $form->setFieldAttribute('published', 'disabled', 'true');
      $form->setFieldAttribute('access', 'disabled', 'true');
    }
    if (empty($id)) {
      $form->setValue('access', null, 0);
    }
    return $form;
  }

  protected function loadFormData() {
    $data = JFactory::getApplication()->getUserState('com_datsogallery.edit.category.data', array());
    if (empty($data)) {
      $data = $this->getItem();
    }
    return $data;
  }

  public function save($data) {
    $app   = JFactory::getApplication();
    $table = $this->getTable();
    $pk    = (!empty($data['id'])) ? $data['id'] : (int) $this->getState($this->getName() . '.id');
    if ($pk > 0) {
      $table->load($pk);
    }
    if ($table->parent_id != $data['parent_id'] || $data['id'] == 0) {
      $table->setLocation($data['parent_id'], 'last-child');
    }
    if (!$table->bind($data)) {
      $this->setError($table->getError());
      return false;
    }
    if (!$table->check()) {
      $this->setError($table->getError());
      return false;
    }
    if (!$table->store()) {
      $this->setError($table->getError());
      return false;
    }
    if (!$table->rebuildPath($table->id)) {
      $this->setError($table->getError());
      return false;
    }
    if (!$table->rebuild($table->id, $table->lft, $table->level, $table->path)) {
      $this->setError($table->getError());
      return false;
    }
    $this->setState($this->getName() . '.id', $table->id);
    $this->cleanCache();
    return true;
  }

  public function delete(&$pks) {
    $pks   = (array) $pks;
    $table = $this->getTable();
    foreach ($pks as $pk) {
      if (!$table->delete($pk)) {
        $this->setError($table->getError());
        return false;
      }
    }
    return true;
  }

  public function publish(&$pks, $value = 0) {
    $pks = (array) $pks;
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    $table = $this->getTable();
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_categories');
      $query->set('published = ' . (int) $value);
      $query->where('id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    $table->rebuild();
    $this->cleanCache();
    return true;
  }

  public function approved(&$pks, $value = 0) {
    $pks = (array) $pks;
    JArrayHelper::toInteger($pks);
    if (empty($pks)) {
      $this->setError(JText::_('COM_DATSOGALLERY_NO_ITEM_SELECTED'));
      return false;
    }
    $table = $this->getTable();
    try {
      $db    = $this->getDbo();
      $query = $db->getQuery(true);
      $query->update('#__datsogallery_categories');
      $query->set('approved = ' . (int) $value);
      $query->where('id IN (' . implode(',', $pks) . ')');
      $db->setQuery($query);
      $db->execute();
    } catch (Exception $e) {
      $this->setError($e->getMessage());
      return false;
    }
    $table->reorder();
    $this->cleanCache();
    return true;
  }

  public function rebuild() {
    $table = $this->getTable();
    if (!$table->rebuild()) {
      $this->setError($table->getError());
      return false;
    }
    $this->cleanCache();
    return true;
  }

  protected function cleanCache($group = null, $client_id = 0) {
    parent::cleanCache('com_datsogallery');
    parent::cleanCache('mod_datsogallery_ultimate');
    parent::cleanCache('mod_datsogallery_slideshow');
  }

  protected function batchFlipordering($value, $pks, $contexts) {
    $successful = array();
    $db         = $this->getDbo();
    $query      = $db->getQuery(true);
    foreach ($pks as $id) {
      $query->select('MAX(ordering)')
        ->from('#__datsogallery_images')
        ->where($db->qn('catid') . ' = ' . $db->q($id));
      $db->setQuery($query);
      $max = (int) $db->loadResult();
      $max++;
      $query->clear();
      $query->update('#__datsogallery_images')
        ->set($db->qn('ordering') . ' = ' . $max . ' - ' . $db->qn('ordering'))
        ->where($db->qn('catid') . ' = ' . $db->q($id));
      $db->setQuery($query);
      if ($db->execute()) {
        $successful[] = $id;
      }
    }
    return empty($successful) ? false : $successful;
  }

  protected function batchCopy($value, $pks, $contexts) {
    $type       = new JUcmType;
    $this->type = $type->getTypeByAlias($this->typeAlias);
    $parts      = explode('.', $value);
    $parentId   = (int) ArrayHelper::getValue($parts, 0, 1);
    $db         = $this->getDbo();
    $newIds     = array();
    if ($parentId) {
      if (!$this->table->load($parentId)) {
        if ($error = $this->table->getError()) {
          $this->setError($error);
          return false;
        } else {
          $this->setError(JText::_('JGLOBAL_BATCH_MOVE_PARENT_NOT_FOUND'));
          $parentId = 0;
        }
      }
      if ($parentId == $this->table->getRootId()) {
        $canCreate = $this->user->authorise('core.create', 'com_datsogallery');
      } else {
        $canCreate = $this->user->authorise('core.create', 'com_datsogallery.category.' . $parentId);
      }
      if (!$canCreate) {
        $this->setError(JText::_('COM_DATSOGALLERY_BATCH_CANNOT_CREATE'));
        return false;
      }
    }
    if (empty($parentId)) {
      if (!$parentId = $this->table->getRootId()) {
        $this->setError($db->getErrorMsg());
        return false;
      } elseif (!$this->user->authorise('core.create', 'com_datsogallery')) {
        $this->setError(JText::_('COM_DATSOGALLERY_BATCH_CANNOT_CREATE'));
        return false;
      }
    }
    $parents = array();
    $query   = $db->getQuery(true)
      ->select('COUNT(id)')
      ->from($db->quoteName('#__datsogallery_categories'));
    $db->setQuery($query);
    try
    {
      $count = $db->loadResult();
    } catch (RuntimeException $e) {
      $this->setError($e->getMessage());
      return false;
    }
    while (!empty($pks) && $count > 0) {
      $pk = array_shift($pks);
      $this->table->reset();
      if (!$this->table->load($pk)) {
        if ($error = $this->table->getError()) {
          $this->setError($error);
          return false;
        } else {
          $this->setError(JText::sprintf('JGLOBAL_BATCH_MOVE_ROW_NOT_FOUND', $pk));
          continue;
        }
      }
      $query->clear()
        ->select('id')
        ->from($db->quoteName('#__datsogallery_categories'))
        ->where('lft > ' . (int) $this->table->lft)
        ->where('rgt < ' . (int) $this->table->rgt);
      $db->setQuery($query);
      $childIds = $db->loadColumn();
      foreach ($childIds as $childId) {
        if (!in_array($childId, $pks)) {
          $pks[] = $childId;
        }
      }
      $oldId                  = $this->table->id;
      $oldParentId            = $this->table->parent_id;
      $this->table->id        = 0;
      $this->table->parent_id = isset($parents[$oldParentId]) ? $parents[$oldParentId] : $parentId;
      $this->table->setLocation($this->table->parent_id, 'last-child');
      $this->table->level     = null;
      $this->table->lft       = null;
      $this->table->rgt       = null;
      $this->table->published = 0;
      if (!$this->table->store()) {
        $this->setError($this->table->getError());
        return false;
      }
      $newId           = $this->table->get('id');
      $newIds[$pk]     = $newId;
      $parents[$oldId] = $this->table->id;
      $count--;
    }
    if (!$this->table->rebuild()) {
      $this->setError($this->table->getError());
      return false;
    }
    if (!$this->table->rebuildPath($this->table->id)) {
      $this->setError($this->table->getError());
      return false;
    }
    return $newIds;
  }

  protected function batchMove($value, $pks, $contexts) {
    $parentId   = (int) $value;
    $type       = new JUcmType;
    $this->type = $type->getTypeByAlias($this->typeAlias);
    $db         = $this->getDbo();
    $query      = $db->getQuery(true);
    if ($parentId) {
      if (!$this->table->load($parentId)) {
        if ($error = $this->table->getError()) {
          $this->setError($error);
          return false;
        } else {
          $this->setError(JText::_('JGLOBAL_BATCH_MOVE_PARENT_NOT_FOUND'));
          $parentId = 0;
        }
      }
      if ($parentId == $this->table->getRootId()) {
        $canCreate = $this->user->authorise('core.create', 'com_datsogallery');
      } else {
        $canCreate = $this->user->authorise('core.create', 'com_datsogallery.category.' . $parentId);
      }

      if (!$canCreate) {
        $this->setError(JText::_('COM_DATSOGALLERY_BATCH_CANNOT_CREATE'));
        return false;
      }
      foreach ($pks as $pk) {
        if (!$this->user->authorise('core.edit', 'com_datsogallery.category.' . $pk)) {
          $this->setError(JText::_('COM_DATSOGALLERY_BATCH_CANNOT_EDIT'));
          return false;
        }
      }
    }
    $children = array();
    foreach ($pks as $pk) {
      if (!$this->table->load($pk)) {
        if ($error = $this->table->getError()) {
          $this->setError($error);

          return false;
        } else {
          $this->setError(JText::sprintf('JGLOBAL_BATCH_MOVE_ROW_NOT_FOUND', $pk));
          continue;
        }
      }
      $this->table->setLocation($parentId, 'last-child');
      if ($parentId != $this->table->parent_id) {
        $query->clear()
          ->select('id')
          ->from($db->quoteName('#__datsogallery_categories'))
          ->where($db->quoteName('lft') . ' BETWEEN ' . (int) $this->table->lft . ' AND ' . (int) $this->table->rgt);
        $db->setQuery($query);
        try
        {
          $children = array_merge($children, (array) $db->loadColumn());
        } catch (RuntimeException $e) {
          $this->setError($e->getMessage());
          return false;
        }
      }
      if (!$this->table->store()) {
        $this->setError($this->table->getError());
        return false;
      }
      if (!$this->table->rebuildPath()) {
        $this->setError($this->table->getError());
        return false;
      }
    }
    if (!empty($children)) {
      $children = array_unique($children);
      $children = ArrayHelper::toInteger($children);
    }
    return true;
  }
}
